package com.tapresearch.ktunitybridge

import android.app.Activity
import android.util.Log
import com.tapresearch.tapsdk.TapInitOptions
import com.tapresearch.tapsdk.TapResearch
import com.tapresearch.tapsdk.callback.TRContentCallback
import com.tapresearch.tapsdk.callback.TRSurveysRefreshedListener
import com.tapresearch.tapsdk.models.QQPayload
import com.tapresearch.tapsdk.models.TRBonusBarProgress
import com.tapresearch.tapsdk.models.TRBonusTier
import com.tapresearch.tapsdk.models.TRError
import com.tapresearch.tapsdk.models.TRGrantBoostResponse
import com.tapresearch.tapsdk.models.TRPlacementDetails
import com.tapresearch.tapsdk.models.TRReward
import com.tapresearch.tapsdk.models.TRSurvey
import com.unity3d.player.UnityPlayer
import kotlinx.serialization.InternalSerializationApi
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import org.json.JSONArray
import org.json.JSONObject

@OptIn(InternalSerializationApi::class)
class TRUnityBridge {

    companion object {
        private const val LOG_TAG = "TRLog_KtBridge"

        @JvmStatic
        fun initialize(apiToken: String, userIdentifier: String, activity: Activity) {
            TapResearch.initialize(
                apiToken = apiToken,
                userIdentifier = userIdentifier,
                activity = activity,
                errorCallback = { error -> Companion.onTapResearchDidError(error) },
                sdkReadyCallback = { Companion.onTapResearchSdkReady() },
                rewardCallback = null,
                qqDataCallback = null,
            )
        }

        @JvmStatic
        fun initializeWithUserAttributes(apiToken: String, userIdentifier: String, userAttributes: String, clearPreviousAttributes: Boolean, activity: Activity) {

            val data = fromJson(userAttributes)
            val initOptions = TapInitOptions(userAttributes = data, clearPreviousAttributes = clearPreviousAttributes)
            TapResearch.initialize(
                apiToken = apiToken,
                userIdentifier = userIdentifier,
                activity = activity,
                initOptions = initOptions,
                errorCallback = { error -> Companion.onTapResearchDidError(error) },
                sdkReadyCallback = { Companion.onTapResearchSdkReady() },
                rewardCallback = null,
                qqDataCallback = null,
            )
        }

        @JvmStatic
        fun updateCurrentUser(userIdentifier: String) {
            TapResearch.setUserIdentifier(
                userIdentifier,
            )
        }

        @JvmStatic
        fun sendUserAttributes(userAttributes: String) {
            val data = fromJson(userAttributes)
            data?.let {
                TapResearch.sendUserAttributes(
                    userAttributes = it,
                    clearPreviousAttributes = false,
                    errorCallback = { error -> Companion.onTapResearchDidError(error) })
            }
            Log.d(LOG_TAG, "userAttributes: $data")
        }

        @JvmStatic
        fun sendUserAttributes(userAttributes: String, clearPreviousAttributes: Boolean) {
            val data = fromJson(userAttributes)
            data?.let {
                TapResearch.sendUserAttributes(
                    userAttributes = it,
                    clearPreviousAttributes = clearPreviousAttributes,
                    errorCallback = { error -> Companion.onTapResearchDidError(error) })
            }
            Log.d(LOG_TAG, "userAttributes: $data")
        }

        @JvmStatic
        fun canShowContent(placementTag: String): Boolean {
            return TapResearch.canShowContentForPlacement(
                placementTag,
            ) { error -> Companion.onTapResearchDidError(error) }
        }

        @JvmStatic
        fun showContent(placementTag: String) {
            TapResearch.showContentForPlacement(
                tag = placementTag,
                contentListener = object : TRContentCallback {
                    override fun onTapResearchContentShown(placementTag: String) {
                        Companion.onTapResearchContentShown(placementTag)
                    }

                    override fun onTapResearchContentDismissed(placementTag: String) {
                        Companion.onTapResearchDidDismiss(placementTag)
                    }
                },
            ) { error -> Companion.onTapResearchDidError(error) }
        }

        @JvmStatic
        fun showContent(placementTag: String, jsonCustomParameters: String?) {
            if (jsonCustomParameters == null) {
                Log.e(LOG_TAG, "jsonCustomParameters is null")
                return
            }

            val customParameters = fromJson(jsonCustomParameters)

            Log.d(LOG_TAG, "ShowContentWithParams: $customParameters")

            TapResearch.showContentForPlacement(
                tag = placementTag,
                customParameters = customParameters,
                contentListener = object : TRContentCallback {
                    override fun onTapResearchContentShown(placementTag: String) {
                        Companion.onTapResearchContentShown(placementTag)
                    }

                    override fun onTapResearchContentDismissed(placementTag: String) {
                        Companion.onTapResearchDidDismiss(placementTag)
                    }
                },
            ) { error -> Companion.onTapResearchDidError(error) }
        }

        @JvmStatic
        fun isReady(): Boolean {
            return TapResearch.isReady()
        }

        @JvmStatic
        fun grantBoost(tag: String) {
            TapResearch.grantBoost(
                tag,
            ) { grantBoostResponse ->
                onGrantBoostResponse(grantBoostResponse)
            }
        }

        private fun onGrantBoostResponse(grantBoostResponse: TRGrantBoostResponse) {
            UnityPlayer.UnitySendMessage(
                "TapResearchSDK",
                "OnTapResearchGrantBoostResponseReceived",
                jsonStringFromGrantBoostResponse(grantBoostResponse),
            )
        }

        private fun onQuickQuestionDataReceived(qqPayload: QQPayload) {
            val qqPayloadString: String = Json.encodeToString(qqPayload)
            UnityPlayer.UnitySendMessage(
                "TapResearchSDK",
                "OnTapResearchQQDataReceived",
                qqPayloadString,
            )
        }

        private fun onTapResearchContentShown(placementTag: String) {
            UnityPlayer.UnitySendMessage(
                "TapResearchSDK",
                "OnTapResearchContentShown",
                placementTag,
            )
        }

        private fun onTapResearchDidDismiss(placementTag: String) {
            UnityPlayer.UnitySendMessage(
                "TapResearchSDK",
                "OnTapResearchContentDismissed",
                placementTag,
            )
        }

        private fun onTapResearchDidError(error: TRError) {
            UnityPlayer.UnitySendMessage("TapResearchSDK",
                "OnTapResearchDidError",
                errorFrom(error).toString())
        }

        private fun onTapResearchDidReceiveRewards(rewards: List<TRReward>) {
            val rewards = jsonArrayFromRewards(rewards)
            UnityPlayer.UnitySendMessage(
                "TapResearchSDK",
                "OnTapResearchDidReceiveRewards",
                rewards,
            )
            Log.d(LOG_TAG, "onTapResearchDidReceiveRewards: $rewards")
        }

        private fun onTapResearchSdkReady() {
            UnityPlayer.UnitySendMessage("TapResearchSDK", "OnTapResearchSdkReady", "")
        }

        private fun onTapResearchSurveysRefreshedForPlacement(placementTag: String) {
            UnityPlayer.UnitySendMessage("TapResearchSDK", "OnTapResearchSurveysRefreshedForPlacement", placementTag)
        }

        @JvmStatic
        fun setCallbackEnabled(callbackName: String, enable: Boolean) {
            when (callbackName) {
                "OnTapResearchDidReceiveRewards" -> {
                    if (enable) {
                        TapResearch.setRewardCallback { rewards ->
                            Companion.onTapResearchDidReceiveRewards(rewards)
                        }
                    } else {
                        TapResearch.setRewardCallback(null)
                    }
                }
                "OnTapResearchQQDataReceived" -> {
                    if (enable) {
                        TapResearch.setQuickQuestionCallback { data ->
                            Companion.onQuickQuestionDataReceived(data)
                        }
                    } else {
                        TapResearch.setQuickQuestionCallback(null)
                    }
                }
                "OnTapResearchSurveysRefreshed" -> {
                    if (enable) {
                        TapResearch.setSurveysRefreshedListener(object : TRSurveysRefreshedListener{
                            override fun onSurveysRefreshedForPlacement(placementTag: String) {
                                Companion.onTapResearchSurveysRefreshedForPlacement(placementTag)
                            }
                        })
                    } else {
                        TapResearch.setSurveysRefreshedListener(null)
                    }
                }
                else -> {
                    // not defined
                }
            }
        }

        @JvmStatic
        fun hasSurveysForPlacement(placementTag: String): Boolean {
            return TapResearch.hasSurveysForPlacement(placementTag) {
                Companion.onTapResearchDidError(it)
            }
        }

        @JvmStatic
        fun getSurveysForPlacement(placementTag: String): String {
            val surveys = TapResearch.getSurveysForPlacement(placementTag) {
                Companion.onTapResearchDidError(it)
            }
            return jsonArrayFromSurveys(surveys)
        }

        @JvmStatic
        fun showSurveyForPlacement(placementTag: String, surveyId: String, jsonCustomParameters: String?) {
            if (jsonCustomParameters == null) {
                Log.e(LOG_TAG, "jsonCustomParameters is null")
                return
            }

            val customParameters = fromJson(jsonCustomParameters)

            Log.d(LOG_TAG, "showSurveyForPlacement: $customParameters")

            TapResearch.showSurveyForPlacement(
                placementTag = placementTag,
                surveyId = surveyId,
                customParameters = customParameters,
                contentListener = object : TRContentCallback {
                    override fun onTapResearchContentShown(placementTag: String) {
                        Companion.onTapResearchContentShown(placementTag)
                    }

                    override fun onTapResearchContentDismissed(placementTag: String) {
                        Companion.onTapResearchDidDismiss(placementTag)
                    }
                },
            ) { error -> Companion.onTapResearchDidError(error) }
        }

        @JvmStatic
        fun showSurveyForPlacement(placementTag: String, surveyId: String) {

            TapResearch.showSurveyForPlacement(
                placementTag = placementTag,
                surveyId = surveyId,
                customParameters = null,
                contentListener = object : TRContentCallback {
                    override fun onTapResearchContentShown(placementTag: String) {
                        Companion.onTapResearchContentShown(placementTag)
                    }

                    override fun onTapResearchContentDismissed(placementTag: String) {
                        Companion.onTapResearchDidDismiss(placementTag)
                    }
                },
            ) { error -> Companion.onTapResearchDidError(error) }
        }

        @JvmStatic
        fun getPlacementDetails(placementTag: String): String? {
            val placementDetails = TapResearch.getPlacementDetails(placementTag)
            return placementDetailsFrom(placementDetails)
        }

        internal fun fromJson(jsonString: String?): HashMap<String,Any>? {
            if (jsonString.isNullOrBlank()) {
                return null
            }
            val hashMap = HashMap<String, Any>()
            try {
                val jsonObject = JSONObject(jsonString)
                for (key in jsonObject.keys()) {
                    hashMap[key] = jsonObject.get(key)
                }
            } catch (e: Exception) {
                Log.e(LOG_TAG,"fromJson failed",e)
            }
            return hashMap
        }

        internal fun jsonArrayFromRewards(rewards: List<TRReward>): String {
            val jsonArray = JSONArray()
            for (reward in rewards) {
                try {
                    val jsonObject = JSONObject()
                    jsonObject.put("transactionIdentifier", reward.transactionIdentifier)
                    jsonObject.put("placementIdentifier", reward.placementIdentifier)
                    jsonObject.put("currencyName", reward.currencyName)
                    jsonObject.put("rewardAmount", reward.rewardAmount?.toInt())
                    jsonObject.put("payoutEvent", reward.payoutEventType)
                    jsonObject.put("placementTag", reward.placementTag)
                    jsonArray.put(jsonObject)
                } catch (exception: Exception) {
                    Log.e(LOG_TAG, "jsonArrayFromRewards", exception)
                }
            }
            return jsonArray.toString()
        }

        internal fun jsonArrayFromSurveys(surveys: List<TRSurvey>?): String {
            val jsonArray = JSONArray()
            surveys?.let {
                for (survey in it) {
                    try {
                        val jsonObject = JSONObject()
                        jsonObject.put("surveyIdentifier", survey.surveyId)
                        jsonObject.put("lengthInMinutes", survey.lengthInMinutes)
                        jsonObject.put("rewardAmount", survey.rewardAmount)
                        jsonObject.put("currencyName", survey.currencyName)
                        jsonObject.put("isSale", survey.isSale ?: false)
                        jsonObject.put("saleEndDate", survey.saleEndDate)
                        jsonObject.put("preSaleRewardAmount", survey.preSaleRewardAmount)
                        jsonObject.put("saleMultiplier", survey.saleMultiplier)
                        jsonObject.put("isHotTile", survey.isHotTile ?: false)
                        jsonObject.put("category", survey.category)
                        jsonArray.put(jsonObject)
                    } catch (exception: Exception) {
                        Log.e(LOG_TAG, "jsonArrayFromSurveys", exception)
                    }
                }
            }
            return jsonArray.toString()
        }

        internal fun placementDetailsFrom(placementDetails: TRPlacementDetails?): String? {
            try {
                placementDetails?.apply {
                    val jsonObject = JSONObject()
                    jsonObject.put("name", name)
                    jsonObject.put("contentType", contentType)
                    jsonObject.put("currencyName", currencyName)
                    jsonObject.put("isSale", isSale)
                    jsonObject.put("saleType", saleType)
                    jsonObject.put("saleEndDate", saleEndDate)
                    jsonObject.put("saleMultiplier", saleMultiplier)
                    jsonObject.put("saleDisplayName", saleDisplayName)
                    jsonObject.put("saleTag", saleTag)
                    bonusBarProgress?.let { bonusBarProgress ->
                        jsonObject.put("bonusBarProgress", bonusBarProgressFrom(bonusBarProgress))
                    }
                    return jsonObject.toString()
                }
            } catch (exception: Exception) {
                Log.e(LOG_TAG, "jsonStringFromPlacementDetails", exception)
            }
            return ""
        }

        internal fun bonusBarProgressFrom(bonusBarProgress: TRBonusBarProgress): JSONObject {
            val jsonObject = JSONObject()
            jsonObject.put("isActive", bonusBarProgress.isActive)
            jsonObject.put("currentCompletes", bonusBarProgress.currentCompletes)
            jsonObject.put("bonusWindowEndAt", bonusBarProgress.bonusWindowEndAt)
            bonusBarProgress.error?.let { error ->
                jsonObject.put("error", errorFrom(error))
            }
            bonusBarProgress.bonusTiers?.let { bonusTiers ->
                jsonObject.put("bonusTiers", bonusTiersFrom(bonusTiers))
            }
            return jsonObject
        }

        internal fun bonusTiersFrom(list: List<TRBonusTier>) : JSONArray {
            val jsonArray = JSONArray()
            list.forEach { bonusTier ->
                val jsonObject = JSONObject()
                jsonObject.put("tierNumber", bonusTier.tierNumber)
                jsonObject.put("completesNeeded", bonusTier.completesNeeded)
                jsonObject.put("rewardAmount", bonusTier.rewardAmount)
                jsonObject.put("status", bonusTier.status)
                jsonArray.put(jsonObject)
            }
            return jsonArray
        }

        internal fun errorFrom(error: TRError): JSONObject {
            val jsonObject = JSONObject()
            jsonObject.put("errorCode", error.code)
            jsonObject.put("errorDescription", error.description)
            jsonObject.put("message", error.description)
            return jsonObject
        }

        internal fun jsonStringFromGrantBoostResponse(grantBoostResponse: TRGrantBoostResponse): String {
            val grantBoostJSONObject = JSONObject(Json.encodeToString(grantBoostResponse))
            grantBoostJSONObject.put("boostTag", grantBoostResponse.boostTag)
            grantBoostResponse.error?.let {
                grantBoostJSONObject.put("error", errorFrom(it))
            }
            return grantBoostJSONObject.toString()
        }
    }
}
