package com.tapresearch.tapsdk

import android.app.Activity
import android.app.Application
import android.content.Context
import com.tapresearch.tapsdk.TapResearch.getPlacementDetails
import com.tapresearch.tapsdk.TapResearch.getSurveysForPlacement
import com.tapresearch.tapsdk.TapResearch.grantBoost
import com.tapresearch.tapsdk.TapResearch.initialize
import com.tapresearch.tapsdk.TapResearch.showContentForPlacement
import com.tapresearch.tapsdk.TapResearch.showSurveyForPlacement
import com.tapresearch.tapsdk.callback.TRContentCallback
import com.tapresearch.tapsdk.callback.TRErrorCallback
import com.tapresearch.tapsdk.callback.TRGrantBoostResponseListener
import com.tapresearch.tapsdk.callback.TRQQDataCallback
import com.tapresearch.tapsdk.callback.TRRewardCallback
import com.tapresearch.tapsdk.callback.TRSdkReadyCallback
import com.tapresearch.tapsdk.callback.TRSurveysRefreshedListener
import com.tapresearch.tapsdk.internal.TapResearchInternal
import com.tapresearch.tapsdk.models.TRPlacementDetails
import com.tapresearch.tapsdk.models.TRSurvey
import kotlinx.serialization.InternalSerializationApi

/**
 * User attributes to send to TapResearch for targeting purposes when initializing the SDK.
 *
 * @property userAttributes Map of user attributes to send to TapResearch for targeting purposes.
 * @property clearPreviousAttributes Pass true to clear any previously sent user attributes; false otherwise
 * @constructor Create empty Tap init options
 *
 * @see TapResearch.initialize
 */
data class TapInitOptions(
    val userAttributes: HashMap<String, Any>? = null,
    val clearPreviousAttributes: Boolean? = false,
)

/**
 * TapResearch - All survey api are contained within this object.  Be sure to call initialize()
 * before any other method, such as showContentForPlacement().
 *
 *
 * @see initialize
 * @see showContentForPlacement
 */
object TapResearch {

    /**
     * Sets the reward listener.
     *
     * @param rewardCallback Listens for reward events; such as when a survey is completed and
     * after the survey wall is closed.  Pass null to disable listening.
     *
     */
    fun setRewardCallback(
        rewardCallback: TRRewardCallback? = null,
    ) {
        TapResearchInternal.setRewardCallback(rewardCallback)
    }

    /**
     * Sets the quick question listener.
     *
     * @param quickQuestionCallback Listens for quick question events; such as when a quick
     * question has been answered along with question details.  Pass null to disable listening.
     *
     */
    fun setQuickQuestionCallback(
        quickQuestionCallback: TRQQDataCallback? = null,
    ) {
        TapResearchInternal.setQuickQuestionCallback(quickQuestionCallback)
    }

    /**
     * Initialize the TapResearch SDK.  Must be called before most TapResearch api.  Does not require
     * context or activity.
     *
     * @param apiToken The API Token; provided by TapResearch team.
     * @param userIdentifier The unique user identifier; can be set again by calling setUserIdentifier.
     * @param rewardCallback Optional reward callback; can also call setRewardCallback
     * @param errorCallback Mandatory error callback; e.g. initialization errors.
     * @param sdkReadyCallback Mandatory SDK ready callback; let's you know the SDK is ready for use.
     * @param qqDataCallback Optional Quick Question data callback; also see setQuickQuestionCallback.
     * @param initOptions optional TapInitOptions.  Contains optional targeting parameters.
     *
     */
    fun initialize(
        apiToken: String,
        userIdentifier: String,
        rewardCallback: TRRewardCallback? = null,
        errorCallback: TRErrorCallback,
        sdkReadyCallback: TRSdkReadyCallback,
        qqDataCallback: TRQQDataCallback? = null,
        initOptions: TapInitOptions? = null,
    ) {
        TapResearchInternal.initialize(
            apiToken = apiToken,
            userIdentifier = userIdentifier,
            rewardCallback = rewardCallback,
            errorCallback = errorCallback,
            sdkReadyCallback = sdkReadyCallback,
            qqDataCallback = qqDataCallback,
            initOptions = initOptions,
        )
    }

    /**
     * Initialize the TapResearch SDK.  Must be called before most TapResearch api.
     *
     * @param apiToken The API Token; provided by TapResearch team.
     * @param userIdentifier The unique user identifier; can be set again by calling setUserIdentifier.
     * @param context Activity or application context
     * @param rewardCallback Optional reward callback; can also call setRewardCallback
     * @param errorCallback Mandatory error callback; e.g. initialization errors.
     * @param sdkReadyCallback Mandatory SDK ready callback; let's you know the SDK is ready for use.
     * @param qqDataCallback Optional Quick Question data callback; also see setQuickQuestionCallback.
     * @param initOptions optional TapInitOptions.  Contains optional targeting parameters.
     *
     */
    fun initialize(
        apiToken: String,
        userIdentifier: String,
        context: Context,
        rewardCallback: TRRewardCallback? = null,
        errorCallback: TRErrorCallback,
        sdkReadyCallback: TRSdkReadyCallback,
        qqDataCallback: TRQQDataCallback? = null,
        initOptions: TapInitOptions? = null,
    ) {
        TapResearchInternal.initialize(
            apiToken = apiToken,
            userIdentifier = userIdentifier,
            context = context,
            rewardCallback = rewardCallback,
            errorCallback = errorCallback,
            sdkReadyCallback = sdkReadyCallback,
            qqDataCallback = qqDataCallback,
            initOptions = initOptions,
        )
    }

    /**
     * Initialize the TapResearch SDK.  Must be called before most TapResearch api.
     *
     * @param apiToken The API Token; provided by TapResearch team.
     * @param userIdentifier The unique user identifier; can be set again by calling setUserIdentifier.
     * @param activity Activity; application Context can also be used.
     * @param rewardCallback Optional reward callback; can also call SetRewardCallback
     * @param errorCallback Mandatory error callback; e.g. initialization errors.
     * @param sdkReadyCallback Mandatory SDK ready callback; let's you know the SDK is ready for use.
     * @param qqDataCallback Optional Quick Question data callback; also see setQuickQuestionCallback.
     * @param initOptions Optional TapInitOptions.  Contains optional targeting parameters.
     *
     */
    fun initialize(
        apiToken: String,
        userIdentifier: String,
        activity: Activity,
        rewardCallback: TRRewardCallback? = null,
        errorCallback: TRErrorCallback,
        sdkReadyCallback: TRSdkReadyCallback,
        qqDataCallback: TRQQDataCallback? = null,
        initOptions: TapInitOptions? = null,
    ) {
        TapResearchInternal.initialize(
            apiToken = apiToken,
            userIdentifier = userIdentifier,
            activity = activity,
            rewardCallback = rewardCallback,
            errorCallback = errorCallback,
            sdkReadyCallback = sdkReadyCallback,
            qqDataCallback = qqDataCallback,
            initOptions = initOptions,
        )
    }

    /**
     * Sets the unique user identifier in TapResearch.  Can cause more placements to become
     * available.
     *
     * @param userIdentifier The unique user identifier.
     */
    fun setUserIdentifier(userIdentifier: String) {
        TapResearchInternal.setUserIdentifier(userIdentifier)
    }

    /**
     * Immediately indicates whether survey wall content can be shown for the given placement.
     *
     * @param tag The placement tag string.  E.g. "home-screen" or "earn-center"
     * @param errorCallback Listens for any errors that might occur during this call.
     * @return True if content for the given placement can be shown.  False otherwise.
     */
    fun canShowContentForPlacement(tag: String, errorCallback: TRErrorCallback): Boolean {
        return TapResearchInternal.canShowContentForPlacement(tag, errorCallback)
    }

    /**
     * Sends user attributes to TapResearch.
     *
     * @param userAttributes The user attributes map that contains your targeting parameters.
     * @param clearPreviousAttributes If true, will clear the previously sent user attributes.
     * @param errorCallback Listens for any errors that might occur during this call.
     *
     */
    fun sendUserAttributes(
        userAttributes: HashMap<String, Any>,
        clearPreviousAttributes: Boolean? = false,
        errorCallback: TRErrorCallback,
    ) {
        TapResearchInternal.sendUserAttributes(
            userAttributes = userAttributes,
            clearPreviousAttributes = clearPreviousAttributes,
            errorCallback = errorCallback,
        )
    }

    /**
     * Show TapResearch content activity for a given placement tag.
     *
     * @param tag The placement tag string.  E.g. "home-screen" or "earn-center"
     * @param contentListener Listens for content events such as content shown and dismissed.
     * @param customParameters Optional custom targeting parameters.
     * @param errorListener Listens for any errors that might occur during this call.
     */
    fun showContentForPlacement(
        tag: String,
        contentListener: TRContentCallback? = null,
        customParameters: HashMap<String, Any>? = null,
        errorListener: TRErrorCallback,
    ) {
        TapResearchInternal.showContentForPlacement(
            tag = tag,
            contentListener = contentListener,
            customParameters = customParameters,
            errorListener = errorListener,
        )
    }

    /**
     * Show TapResearch content activity for a given placement tag.
     *
     * @param tag The placement tag string.  E.g. "home-screen" or "earn-center"
     * @param contentCallback Listens for content shown and dismissed events.
     * @param customParameters Optional custom targeting parameters.
     * @param errorCallback Listens for any errors that might occur during this call.
     * @param optionalParentActivity Specify optional parent activity.  Not recommended unless
     * you have a unique situation where you want to show a survey from the device's lock screen.
     */
    fun showContentForPlacement(
        tag: String,
        contentCallback: TRContentCallback? = null,
        customParameters: HashMap<String, Any>? = null,
        errorCallback: TRErrorCallback,
        optionalParentActivity: Activity? = null,
    ) {
        TapResearchInternal.showContentForPlacement(
            tag = tag,
            contentCallback = contentCallback,
            customParameters = customParameters,
            errorCallback = errorCallback,
            optionalParentActivity = optionalParentActivity,
        )
    }

    @Deprecated(
        message = "No longer need application parameter",
        replaceWith = ReplaceWith("showContentForPlacement(tag, contentCallback, customParameters, errorCallback)"),
        level = DeprecationLevel.WARNING,
    )
    fun showContentForPlacement(
        tag: String,
        application: Application,
        contentListener: TRContentCallback? = null,
        customParameters: HashMap<String, Any>? = null,
        errorListener: TRErrorCallback,
    ) {
        TapResearchInternal.showContentForPlacement(
            tag = tag,
            application = application,
            contentListener = contentListener,
            customParameters = customParameters,
            errorListener = errorListener,
        )
    }

    /**
     * Optional!  Only invoke when you are sure the user wants to quit the entire app
     * and remove all internal caching.  Not detrimental if not used.
     * Do not call from Activity.onDestroy().  After calling, the next initialize call
     * will be slow.  Does not improve memory consumption.
     */
    fun quitAndCleanup() {
        TapResearchInternal.quitAndCleanup()
    }

    // Comment out of the official public SDK - only for diagnostic and hardening purposes!
    /*
    fun kill(inSeconds: Int? = 5, doSetWebViewToNull: Boolean = false, doCrashWebView: Boolean = false) {
        TapResearchInternal.kill(inSeconds, doSetWebViewToNull, doCrashWebView)
    }
    */

    /**
     * Immediately returns true if TapResearch SDK was successfully initialized and is ready to be used.
     * False, otherwise.
     */
    fun isReady(): Boolean {
        return TapResearchInternal.isReady()
    }

    /**
     * If return true, indicates TapResearch SDK is currently in the process
     * of initializing.
     * If false, indicates SDK initialization process has either not started
     * yet OR the process has failed due to bad network or some other issue.
     */
    internal fun isInitializing(): Boolean {
        return TapResearchInternal.isInitializing()
    }

    /**
     * Set surveys refreshed listener.
     *
     * @param surveysRefreshedListener Listens for when native preview wall placements are
     * refreshed.  The refreshed placement tag can then be queried by calling getSurveysForPlacement.
     * Passing null will un-listen and could be called in onPause() lifecycle events, if desired.
     * @see getSurveysForPlacement
     *
     */
    fun setSurveysRefreshedListener(surveysRefreshedListener: TRSurveysRefreshedListener?) {
        TapResearchInternal.setSurveysRefreshedListener(surveysRefreshedListener)
    }

    /**
     * Immediately indicates if native preview wall surveys are available for a given placement.
     *
     * @param placementTag The placement tag string.  E.g. "home-screen" or "earn-center"
     * @param errorListener Listens for any errors that might occur during this call.
     * @return true if native surveys are available for given placement.
     *
     * @see getSurveysForPlacement
     */
    fun hasSurveysForPlacement(placementTag: String,
                               errorListener: TRErrorCallback): Boolean {
        return TapResearchInternal.hasSurveysForPlacement(
            placementTag = placementTag,
            errorListener = errorListener,
        )
    }

    /**
     * Immediately returns a list of cached native preview wall surveys for the given placement.
     *
     * @param placementTag - The placement tag string.  E.g. "home-screen" or "earn-center"
     * @param errorListener - Listens for any errors that might occur during this call.
     * @return an immediate list of cached native preview wall surveys.  null, if none are available.
     *
     * @see showSurveyForPlacement
     */
    fun getSurveysForPlacement(placementTag: String,
                               errorListener: TRErrorCallback): List<TRSurvey>? {
        return TapResearchInternal.getSurveysForPlacement(
            placementTag = placementTag,
            errorListener = errorListener,
        )
    }

    /**
     * Show a native preview wall survey for the given survey id AND placement.  Survey id parameter
     * was obtained via getSurveysForPlacement.
     *
     * @param placementTag The placement tag string.  E.g. "home-screen" or "earn-center"
     * @param surveyId Survey id of the native preview wall survey to be shown.
     * @param customParameters Optional custom targeting parameters.
     * @param contentListener Optional; Listens for content shown and dismiss events.
     * @param errorListener Listens for any errors that might occur during this call.
     *
     * @see getSurveysForPlacement
     */
    fun showSurveyForPlacement(
                                placementTag: String,
                                surveyId: String,
                                customParameters: HashMap<String, Any>? = null,
                                contentListener: TRContentCallback? = null,
                                errorListener: TRErrorCallback,
                               ) {
        TapResearchInternal.showSurveyForPlacement(
            placementTag = placementTag,
            surveyId = surveyId,
            customParameters = customParameters,
            contentListener = contentListener,
            errorListener = errorListener,
        )
    }

    /**
     * Grant boost - Allow publishers to grant a player-level boost by tag.
     *
     * @param boostTag - Provided by TapResearch team; contact your representative.
     * @param grantBoostResponseListener - Optional. Listens for response to grant boost request.
     *                                     If success, it would be a good time to call getPlacementDetails
     *                                     since the details would be updated with boosted data.
     * @see getPlacementDetails
     */
    fun grantBoost(boostTag: String, grantBoostResponseListener: TRGrantBoostResponseListener? = null) {
        TapResearchInternal.grantBoost(boostTag, grantBoostResponseListener)
    }

    /**
     * Get placement details
     *
     * @param placementTag
     * @param errorListener - such as invalid placement tag.
     * @return - placement details
     *
     * @see grantBoost
     */
    @OptIn(InternalSerializationApi::class)
    fun getPlacementDetails(placementTag: String, errorListener: TRErrorCallback? = null): TRPlacementDetails? {
        return TapResearchInternal.getPlacementDetails(placementTag, errorListener)
    }

}

