package com.tapresearch.tapsdk.models

import androidx.annotation.Keep
import kotlinx.serialization.InternalSerializationApi
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.builtins.serializer
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.JsonTransformingSerializer
import kotlinx.serialization.json.int
import kotlinx.serialization.json.jsonPrimitive

@Serializable
enum class PayoutTypes(val value: Int) {
    @SerialName("profile_reward") PROFILE_REWARD(0),
    @SerialName("partial_payout") PARTIAL_PAYOUT(1),
    @SerialName("full_payout") FULL_PAYOUT(3),
    @SerialName("quick_questions_payout") QUICK_QUESTIONS_PAYOUT(9);

    companion object {
        // Using Kotlin's built-in enumValueOf function
        fun fromInt(value: Int) = values().firstOrNull { it.value == value }
    }
}

/**
 * TapResearch Reward
 *
 * @property transactionIdentifier Unique transaction identifier
 * @property placementIdentifier Unique placement identifier; not the placement tag.
 * @property currencyName Currency name
 * @property rewardAmount Reward amount
 * @property payoutEventType Payout event type
 * @property placementTag The placement tag string.  E.g. "home-screen" or "earn-center"
 */
@OptIn(InternalSerializationApi::class) @Serializable
@Keep
data class TRReward(
    @SerialName("transactionIdentifier")
    @Keep
    val transactionIdentifier: String? = null,

    @SerialName("placementIdentifier")
    @Keep
    val placementIdentifier: String? = null,

    @SerialName("currencyName")
    @Keep
    val currencyName: String? = null,

    @SerialName("rewardAmount")
    @Keep
    val rewardAmount: Float? = null,

    @Serializable(with = PayoutEventTypeSerializer::class)
    @SerialName("payoutEventType")
    @Keep
    val payoutEventType: String? = null,

    @SerialName("placementTag")
    @Keep
    val placementTag: String? = null,
)

object PayoutEventTypeSerializer : JsonTransformingSerializer<String>(String.serializer()) {
    // Converts JSON integer value to PayoutTypes enum
    override fun transformDeserialize(element: JsonElement): JsonElement {
        val intValue = element.jsonPrimitive.int
        val payoutType = PayoutTypes.fromInt(intValue)?.name
        return JsonPrimitive(payoutType)
    }

    // Converts PayoutTypes enum to its corresponding JSON integer value
    override fun transformSerialize(element: JsonElement): JsonElement {
        val stringValue = element.jsonPrimitive.content
        val payoutType = PayoutTypes.valueOf(stringValue.uppercase()).value
        return JsonPrimitive(payoutType)
    }
}
