package com.tapresearch.tapsdk.webview

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Bitmap
import android.os.Build
import android.os.Message
import android.webkit.ConsoleMessage
import android.webkit.RenderProcessGoneDetail
import android.webkit.WebChromeClient
import android.webkit.WebResourceError
import android.webkit.WebResourceRequest
import android.webkit.WebResourceResponse
import android.webkit.WebSettings
import android.webkit.WebView
import android.webkit.WebViewClient
import com.tapresearch.tapresearchkotlinsdk.BuildConfig
import com.tapresearch.tapsdk.SdkEventDetail
import com.tapresearch.tapsdk.TapResearch
import com.tapresearch.tapsdk.callback.TRContentCallback
import com.tapresearch.tapsdk.callback.TRErrorCallback
import com.tapresearch.tapsdk.callback.TRJSInterface
import com.tapresearch.tapsdk.callback.TRQQDataCallback
import com.tapresearch.tapsdk.callback.TRRewardCallback
import com.tapresearch.tapsdk.callback.TRSdkReadyCallback
import com.tapresearch.tapsdk.models.TRError
import com.tapresearch.tapsdk.models.TRInitPayload
import com.tapresearch.tapsdk.state.EventType
import com.tapresearch.tapsdk.state.LogLevel
import com.tapresearch.tapsdk.state.SdkState
import com.tapresearch.tapsdk.state.SdkStateHolder
import com.tapresearch.tapsdk.state.TRWebViewState
import com.tapresearch.tapsdk.storage.ParameterStorage
import com.tapresearch.tapsdk.storage.PreferenceManager
import com.tapresearch.tapsdk.storage.SdkFeatureToggleStorage
import com.tapresearch.tapsdk.storage.SdkToggles
import com.tapresearch.tapsdk.utils.ConnectionUtils
import com.tapresearch.tapsdk.utils.LogUtils
import com.tapresearch.tapsdk.utils.RemoteEventLogger
import com.tapresearch.tapsdk.utils.TapConstants
import com.tapresearch.tapsdk.utils.TapConstants.REMOTE_LOG_LEVEL
import com.tapresearch.tapsdk.utils.TapConstants.RESTART_TIMER
import com.tapresearch.tapsdk.utils.TapErrorCodes
import com.tapresearch.tapsdk.utils.TimerUtil
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.serialization.InternalSerializationApi
import org.json.JSONObject
import java.util.LinkedList

@InternalSerializationApi
internal class TROrchestrator(private val apiToken: String, private var userIdentifier: String?, private val applicationContext: Context, private var presentationCallback: (configuration: String) -> Unit, private var rewardCallback: TRRewardCallback? = null, private var sdkReadyCallback: TRSdkReadyCallback? = null, internal var contentCallback: TRContentCallback? = null, internal var errorCallback: TRErrorCallback, private var tapDataCallback: TRQQDataCallback? = null, private val isManualInit: Boolean = false, internal val sdkStateHolder: SdkStateHolder) {

    companion object {
        private const val TAG = "TRLog_OR"
    }

    internal data class PendingTransaction(val message: String, val callback: ((String?) -> Unit)?)

    internal var orcaWebView: WebView? = null
    internal var surveyWebView: WebView? = null
    internal var dialogWebView: WebView? = null
    internal var trjsInterface: TRJSInterface? = null
    internal var showContentForPlacementErrorCallback: TRErrorCallback? = null
    internal var sendUserAttributesErrorCallback: TRErrorCallback? = null
    private val pendingTransactions = LinkedList<PendingTransaction>()
    internal var isRestarting = false

    var webViewCallback: ((webViewState: TRWebViewState) -> Unit)? = null

    var webViewMessage: Message? = null

    private val internalWebViewCallback: (webViewState: TRWebViewState) -> Unit = {
        webViewCallback?.invoke(it)
    }

    private val internalContentCallback: TRContentCallback = object : TRContentCallback {
        override fun onTapResearchContentShown(placementTag: String) {
            contentCallback?.onTapResearchContentShown(placementTag)
        }

        override fun onTapResearchContentDismissed(placementTag: String) {
            contentCallback?.onTapResearchContentDismissed(placementTag)
        }
    }

    internal fun storeTag() {
        evaluateJavascript("storeTag('ZGVkOTliOTdlNjBmZTZkNzY4ZjBlMmRhM2IxMjFmZDg=')")
    }

    // This callback is called when the SDK is initialized.
    private val sdkInitCallback: (initPayload: TRInitPayload) -> Unit = { initPayload ->

        if (BuildConfig.DEBUG && TapResearch.doTestSkipSdkInit) {
            // doSkipSdkInitCallback will cause initialization timeout

            // doErrorOnInitialize will cause initialization error
            if (TapResearch.doTestSdkInitError) {
                val errorPayload = JSONObject()
                errorPayload.put("error_code", 100)
                errorPayload.put("message", "intentional test error")
                trjsInterface?.onTapResearchDidError(errorPayload.toString())
            }
        } else {

            // Check if the userIdentifier from the callback is different from the one originally set.
            if (userIdentifier != initPayload.userIdentifier) {
                // Update the userIdentifier with the one from the callback.
                userIdentifier = initPayload.userIdentifier
            }
            sdkStateHolder.state = SdkState.initialized
            storeTag()
            if (isRestarting) {
                TimerUtil.stopTimer(RESTART_TIMER)
                RemoteEventLogger.postEvent(
                    REMOTE_LOG_LEVEL,
                    "TROrchestrator.sdkInitCallback",
                    "pending transactions: $pendingTransactions",
                    null,
                    TapConstants.LOG_CAT_ORCA_REVIVE,
                    TapConstants.LOG_DET_ORCA_REVIVE_SUCCESS,
                )
                isRestarting = false
                synchronized(this@TROrchestrator) {
                    if (pendingTransactions.isNotEmpty()) {
                        // orca was restarted, execute all the pending transactions and clear them out
                        val retryAllowed =
                            SdkFeatureToggleStorage.isEnabled(SdkToggles.RETRY_CALLBACK_ON_RESTART)
                        for (pendingTransaction in pendingTransactions) {
                            LogUtils.internal(
                                TAG,
                                "execute pending transaction: $pendingTransaction"
                            )
                            if (retryAllowed) {
                                evaluateJavascript(
                                    pendingTransaction.message,
                                    pendingTransaction.callback,
                                    false
                                )
                            } else {
                                onEvaluateJavascriptFailed(
                                    pendingTransaction.message,
                                    pendingTransaction.callback,
                                    true
                                )
                            }
                        }
                        pendingTransactions.clear()
                    }
                }
            } else {
                notifyRewardCallbackAvailability(trjsInterface?.rewardCallback)
                notifyQQDataCallbackAvailability(trjsInterface?.qqDataCallback)
                sdkReadyCallback?.onTapResearchSdkReady()
            }
        }
    }

    internal fun resetRewardCallback(
        rewardCallback: TRRewardCallback? = null,
    ) {
        trjsInterface?.resetRewardCallback(rewardCallback)
    }

    internal fun resetQQDataCallback(
        tapDataCallback: TRQQDataCallback? = null,
    ) {
        trjsInterface?.resetQQDataCallback(tapDataCallback)
    }

    internal fun notifyRewardCallbackAvailability(
        rewardCallback: TRRewardCallback? = null,
    ) {
        TapResearch.orchestrator?.evaluateJavascript("setCallbackAvailable('${TapConstants.Callbacks.REWARD}', ${rewardCallback != null})")
    }

    internal fun notifyQQDataCallbackAvailability(
        qqDataCallback: TRQQDataCallback? = null,
    ) {
        TapResearch.orchestrator?.evaluateJavascript("setCallbackAvailable('${TapConstants.Callbacks.QQ_DATA}', ${qqDataCallback != null})")
    }

    internal fun resetCallbacks(
        presentationCallback: (configuration: String) -> Unit,
        rewardCallback: TRRewardCallback? = null,
        sdkReadyCallback: TRSdkReadyCallback? = null,
        contentCallback: TRContentCallback? = null,
        errorCallback: TRErrorCallback,
        tapDataCallback: TRQQDataCallback? = null,
    ) {
        this.presentationCallback = presentationCallback
        this.rewardCallback = rewardCallback
        this.sdkReadyCallback = sdkReadyCallback
        this.contentCallback = contentCallback
        this.errorCallback = errorCallback
        this.tapDataCallback = tapDataCallback
        trjsInterface?.resetCallbacks(
            rewardCallback,
            presentationCallback,
            internalContentCallback,
            internalWebViewCallback,
            sdkInitCallback,
            errorCallback,
            tapDataCallback,
        )
        if (sdkStateHolder.state == SdkState.initialization_failed) {
            if (orcaWebView != null) {
                loadSDKUrl()
            } else {
                initialize()
            }
        }
    }

    private fun initialize(isRevive: Int? = 0) {
        try {
            orcaWebView = WebView(applicationContext)
            orcaWebView?.let {
                configureWebView(it)
                loadSDKUrl(isRevive)
            }
        } catch (t: Throwable) {
            isRestarting = false
            sdkStateHolder.state = SdkState.initialization_failed
            RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TROrchestrator.initialize", "Failed to initialize: ${t.message}")
            errorCallback.onTapResearchDidError(
                TRError(
                    code = TapErrorCodes.SERVICE_UNAVAILABLE.code,
                    description = TapErrorCodes.SERVICE_UNAVAILABLE.errorMessage(),
                ),
            )
        }
    }

    init {
        if (BuildConfig.DEBUG) {
            // try/catch since crashes/fails during unit test run
            try {
                WebView.setWebContentsDebuggingEnabled(true)
            } catch (ignored: Throwable) {}
        }
        initialize()
    }

    internal fun captureSdkEvent(
        errorDetail: SdkEventDetail,
        eventType: EventType,
        logLevel: LogLevel,
        endSession: Boolean = false,
    ) {
        try {
            val errorHash = mutableMapOf(
                "message" to errorDetail.message,
                "class_name" to errorDetail.className,
            )
            val errorJson = (errorHash as Map<*, *>?)?.let { JSONObject(it) }
            val errorMsg = "captureSdkEvent(${eventType.ordinal}, ${logLevel.ordinal}, '${
            errorJson.toString().replace(
                "'",
                "\\'",
            )
            }', $endSession)"
            evaluateJavascript(errorMsg)
        } catch (_: Exception) {}
    }

    /**
     * Only invoke when orca death was detected
     */
    internal fun restart() {
        if (sdkStateHolder.state != SdkState.initializing
            && sdkStateHolder.state != SdkState.not_started_yet
            && ConnectionUtils.isOnline(applicationContext)) {
            LogUtils.internal(TAG, "--- Restarting Orca ---")
            startRestartTimer()
            TapResearch.orcaCanRespond = false
            sdkStateHolder.state = SdkState.initializing
            isRestarting = true
            webViewCallback?.invoke(TRWebViewState.CloseWebView)
            TapResearch.setOrchestratorWebViewCallback(null)
            TapResearch.orchestrator?.webViewMessage = null
            if (orcaWebView != null) {
                loadSDKUrl(1)
            } else {
                initialize(1)
            }
        }
    }

    /**
     * Note: must run within main dispatcher coroutine scope or on main thread.
     */
    internal fun checkOrca(onOrcaIsOk: (Boolean, String) -> Unit) {
        if (orcaWebView == null) {
            onOrcaIsOk(false, "orcaWebView null")
            return
        }
        orcaWebView?.evaluateJavascript("typeof window.appDidBecomeActive") { result ->
            LogUtils.internal(TAG, "checkOrca: $result")
            val statusOk = !result.contains("undefined")
            onOrcaIsOk(statusOk, if (statusOk) "" else "failed orca health-check")
        }
    }

    internal fun evaluateJavascript(message: String, callback: ((String?) -> Unit)? = null, restartWhenFail: Boolean = true) {
        CoroutineScope(Dispatchers.Main).launch {
            checkOrca { isOk, reason ->
                if (isOk) {
                    orcaWebView?.evaluateJavascript(message) { result ->
                        LogUtils.internal(
                            TAG,
                            "Evaluate JS message: $message   Result -> [$result]",
                        )
                        // note: null, true, false for result
                        callback?.invoke(result) // Invoke the callback if it's not null
                    }
                } else {
                    if (sdkStateHolder.state != SdkState.initializing
                        && sdkStateHolder.state != SdkState.not_started_yet) {
                        LogUtils.internal(TAG, "--- Orca has died x_x !!! ---")
                        if (restartWhenFail) {
                            RemoteEventLogger.postEvent(
                                REMOTE_LOG_LEVEL,
                                "TROrchestrator.evaluateJavascript",
                                "$reason. js-message: [$message] callback: [$callback] sdk status: [${sdkStateHolder.state}] restarting orca.",
                                null,
                                TapConstants.LOG_CAT_ORCA_REVIVE,
                                TapConstants.LOG_DET_ORCA_REVIVE_ATTEMPT
                            )
                            synchronized(this@TROrchestrator) {
                                pendingTransactions.add(PendingTransaction(message, callback))
                            }
                            restart()
                        } else {
                            // retry request failed
                            onEvaluateJavascriptFailed(message, callback, false)
                        }
                    }
                }
            }
        }
    }

    private fun onEvaluateJavascriptFailed(message: String, callback: ((String?) -> Unit)? = null, isFirstAndOnlyAttempt: Boolean = true) {
        if (message.startsWith("showContentForPlacement")) {
            showContentForPlacementErrorCallback?.onTapResearchDidError(
                TRError(
                    code = TapErrorCodes.SERVICE_UNAVAILABLE.code,
                    description = TapErrorCodes.SERVICE_UNAVAILABLE.description,
                ),
            )
        } else if (message.startsWith("sendUserAttributes")) {
            sendUserAttributesErrorCallback?.onTapResearchDidError(
                TRError(
                    code = TapErrorCodes.SERVICE_UNAVAILABLE.code,
                    description = TapErrorCodes.SERVICE_UNAVAILABLE.description,
                ),
            )
        }
        if (!isFirstAndOnlyAttempt) {
            RemoteEventLogger.postEvent(
                REMOTE_LOG_LEVEL,
                "TROrchestrator.evaluateJavascript",
                "2nd attempt failed.  message: [$message]  callback: [$callback]  sdk status: [${sdkStateHolder.state}]",
            )
        }
    }

    private fun loadSDKUrl(isRevive: Int? = 0) {
        CoroutineScope(Dispatchers.IO).launch {
            val urlString = buildURL(isRevive)
            withContext(Dispatchers.Main) {
                if (!ConnectionUtils.isOnline(applicationContext)) {
                    errorCallback.onTapResearchDidError(
                        TRError(
                            code = TapErrorCodes.NO_INTERNET_CONNECTION.code,
                            description = TapErrorCodes.NO_INTERNET_CONNECTION.description,
                        )
                    )
                } else {
                    sdkStateHolder.state = SdkState.initializing
                    LogUtils.internal(TAG, "orcaUrl: length: ${urlString.length} $urlString")
                    // LogUtils.internal(TAG, "orcaUrl: url-decoded: ${URLDecoder.decode(urlString, "UTF-8")}")
                    orcaWebView?.loadUrl(urlString)
                }
            }
        }
    }

    @SuppressLint("SetJavaScriptEnabled")
    internal fun configureWebView(orcaWebView: WebView) {
        orcaWebView.settings.apply {
            javaScriptEnabled = true
            cacheMode = WebSettings.LOAD_DEFAULT
            setSupportMultipleWindows(true)
            domStorageEnabled = true
            javaScriptCanOpenWindowsAutomatically = true
        }

        trjsInterface = TRJSInterface(
            rewardCallback,
            presentationCallback,
            internalContentCallback,
            internalWebViewCallback,
            sdkInitCallback,
            errorCallback,
            tapDataCallback,
        )

        trjsInterface?.let {
            orcaWebView.addJavascriptInterface(it, "Android")
        }

        configureWebViewClient()
        configureWebChromeClient()
    }

    private fun configureWebViewClient() {
        orcaWebView?.webViewClient = object : WebViewClient() {
            override fun onRenderProcessGone(view: WebView?, detail: RenderProcessGoneDetail?): Boolean {
                val didCrash = if (Build.VERSION.SDK_INT >= 26) detail?.didCrash() else true
                if (orcaWebView != null) {
                    // only log once! receive multiple when artificially induced.
                    RemoteEventLogger.postEvent(
                        REMOTE_LOG_LEVEL,
                        "TROrchestrator.configureWebViewClient",
                        "orca webview received onRenderProcessGone from system to reclaim resources. didCrash: $didCrash."
                    )
                }
                // set orcaWebView to null, since totally useless at this point
                orcaWebView?.also { webView ->
                    webView.webViewClient = WebViewClient() // don't receive more callbacks, we're done
                    webView.destroy()
                    orcaWebView = null
                }
                return true // The app continues executing.
            }

            override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {
                super.onPageStarted(view, url, favicon)
                LogUtils.i("TROrchestratorWebView", "Load url: $url")
            }

            override fun onReceivedHttpError(
                view: WebView?,
                request: WebResourceRequest?,
                errorResponse: WebResourceResponse?,
            ) {
                super.onReceivedHttpError(view, request, errorResponse)
                LogUtils.e("TROrchestratorWebView", "onReceivedHttpError: $errorResponse")
            }

            override fun onReceivedError(
                view: WebView?,
                request: WebResourceRequest?,
                error: WebResourceError?,
            ) {
                super.onReceivedError(view, request, error)
                val description = error?.description ?: "Unknown error"
                val message = "onReceivedError: $description url: ${request?.url} sdk->[${sdkStateHolder.state}]"
                LogUtils.e("TROrchestrator", message)
                val errorDetail = SdkEventDetail(
                    message = message,
                    className = "TROrchestrator",
                )
                captureSdkEvent(
                    errorDetail,
                    EventType.LOG,
                    LogLevel.ERROR,
                    false,
                )
            }
        }
    }

    private fun configureWebChromeClient() {
        orcaWebView?.webChromeClient = object : WebChromeClient() {
            // var lastErrorTime: Long = 0

            // This is the cool down period for logging JS errors and prevents spamming the logs.
            // val errorCoolDown = 5000 // 5 seconds

            override fun onCreateWindow(
                view: WebView?,
                isDialog: Boolean,
                isUserGesture: Boolean,
                resultMsg: Message?,
            ): Boolean {
                webViewMessage = resultMsg

                return true
            }

            override fun onConsoleMessage(consoleMessage: ConsoleMessage?): Boolean {
                if (BuildConfig.DEBUG && consoleMessage?.message()?.contains("ORCA") == true) {
                    LogUtils.e(TAG, consoleMessage.message())
                }
                return true
            }
//
//                if (consoleMessage?.messageLevel() == ConsoleMessage.MessageLevel.ERROR) {
//                    val currentTime = System.currentTimeMillis()
//                    if (currentTime - lastErrorTime > errorCoolDown) {
//                        lastErrorTime = currentTime
//                        LogUtils.e("WebChromeClient", "JS Error: ${consoleMessage.message()}")
//
//                        // Call your error callback here if you want to propagate JS errors
//                        errorCallback?.onTapResearchDidError(
//                            TRError(
//                                code = TapErrorCodes.UNABLE_TO_DECODE.code,
//                                "JS Error: ${consoleMessage.message()}",
//                            ),
//                        )
//                    }
//                }
//                // Return true if the message is handled by this method. Otherwise, the message will be logged by the system.
//                return true
//            }
        }
    }

    internal fun buildURL(
        isRevive: Int? = 0,
    ): String {
        ParameterStorage.populateFromMain(applicationContext)
        ParameterStorage.populateFromIO(applicationContext)
        ParameterStorage.saveApiToken(apiToken)
        ParameterStorage.saveUserIdentifier(userIdentifier)
        val useHoth = PreferenceManager.getUseHothLegacyPrefs(applicationContext)
        val manualInit = if (isManualInit) 1 else 0
        val currentTimestamp = (System.currentTimeMillis() / 10000)
        val initTimestamp = System.currentTimeMillis()
        val env = if (!useHoth) "" else "&env=hoth"
        val baseUrl = if (!useHoth) BuildConfig.SDK_BASE_URL else "https://sdk-orchestrator-hoth.tapresearch.com"
        return "$baseUrl/?t=${currentTimestamp}/#?api_token=$apiToken&user_identifier=$userIdentifier&manual_init=$manualInit$env&is_revive=$isRevive&init_timestamp=$initTimestamp${ParameterStorage.urlParams()}"
    }

    internal fun onOrcaRestartFailed() {
        isRestarting = false
        sdkStateHolder.state = SdkState.initialization_failed
    }

    private fun startRestartTimer() {
        TimerUtil.startTimer(RESTART_TIMER, 20) {
            if (isRestarting) {
                onOrcaRestartFailed()
                RemoteEventLogger.postEvent(
                    REMOTE_LOG_LEVEL,
                    "TROrchestrator.startRestartTimer",
                    "orca restart timed out",
                    null,
                    TapConstants.LOG_CAT_ORCA_REVIVE,
                    TapConstants.LOG_DET_ORCA_REVIVE_FAIL)
            }
        }
    }
}
