package com.tapresearch.tapsdk.webview

import android.annotation.SuppressLint
import android.app.Activity
import android.app.AlertDialog
import android.content.Context
import android.content.Intent
import android.content.pm.ActivityInfo
import android.content.res.Configuration
import android.content.res.Resources
import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.graphics.drawable.GradientDrawable
import android.net.Uri
import android.net.UrlQuerySanitizer
import android.os.Build
import android.os.Bundle
import android.util.TypedValue
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.webkit.RenderProcessGoneDetail
import android.webkit.URLUtil
import android.webkit.WebResourceError
import android.webkit.WebResourceRequest
import android.webkit.WebView
import android.webkit.WebViewClient
import android.widget.ImageButton
import android.widget.LinearLayout
import android.widget.RelativeLayout
import android.widget.Toolbar
import androidx.core.content.ContextCompat
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import androidx.core.view.updateLayoutParams
import com.tapresearch.tapresearchkotlinsdk.R
import com.tapresearch.tapsdk.SdkEventDetail
import com.tapresearch.tapsdk.TapResearch
import com.tapresearch.tapsdk.WebViewTypeOpen
import com.tapresearch.tapsdk.callback.TRErrorCallback
import com.tapresearch.tapsdk.callback.TRSurveyInterface
import com.tapresearch.tapsdk.models.configuration.TRConfiguration
import com.tapresearch.tapsdk.state.EventType
import com.tapresearch.tapsdk.state.LogLevel
import com.tapresearch.tapsdk.state.TRWebViewState
import com.tapresearch.tapsdk.storage.ParameterStorage
import com.tapresearch.tapsdk.utils.ActivityUtils
import com.tapresearch.tapsdk.utils.DeviceUtils.isLargeScreenDevice
import com.tapresearch.tapsdk.utils.LogUtils
import com.tapresearch.tapsdk.utils.RemoteEventLogger
import com.tapresearch.tapsdk.utils.TapConstants
import com.tapresearch.tapsdk.utils.TapConstants.REMOTE_LOG_LEVEL
import kotlinx.serialization.InternalSerializationApi
import kotlinx.serialization.json.Json
import kotlinx.serialization.decodeFromString

internal class TRWebViewActivity : Activity() {
    private lateinit var config: TRConfiguration
    private var backButtonState: TRWebViewState? = null
    private val json = Json { ignoreUnknownKeys = true }
    private var restore = false
    private var errorCallback = TRErrorCallback {}
    private var onBackPressedCallback: Any? = null
    private val TAG = "TRLog_WVA"

    private fun onBackPress() {
        LogUtils.internal(TAG, "---- Received onBackPress ----")
        if (backButtonState == TRWebViewState.ShowCloseButton) {
            finish()
        } else {
            TapResearch.orchestrator?.surveyWebView?.let { webView ->
                showAbandonDialog(webView)
            }
        }
    }

    private fun checkHasNotch(windowInsets: WindowInsetsCompat) {
        if (Build.VERSION.SDK_INT >= 28) {
            try {
                ParameterStorage.saveHasNotch(this, windowInsets.displayCutout != null)
            }catch (_:Throwable){}
        }
    }

    // As of api 35, status bar and toolbar(or appbar) are not considered part of the system layout
    // so need to manually adjust margins to prevent overlay and cut-off.
    // https://developer.android.com/develop/ui/views/layout/edge-to-edge#kotlin
    private fun adjustEdgeToEdgeMargins(view: View) {
        ViewCompat.setOnApplyWindowInsetsListener(view) { v, windowInsets ->
            checkHasNotch(windowInsets)
            if (Build.VERSION.SDK_INT >= 35) {
                val insets = windowInsets.getInsets(WindowInsetsCompat.Type.systemBars())
                v.updateLayoutParams<ViewGroup.MarginLayoutParams> {
                    topMargin = insets.top
                    leftMargin = insets.left
                    bottomMargin = insets.bottom
                    rightMargin = insets.right
                }
            }
            // Return CONSUMED to prevent window insets from passing down to descendant views
            WindowInsetsCompat.CONSUMED
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        LogUtils.internal(TAG, "onCreate")
        super.onCreate(savedInstanceState)

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            LogUtils.internal(TAG, "---- Register onBackInvoked callback ----")
            onBackPressedCallback = android.window.OnBackInvokedCallback {
                LogUtils.internal(TAG, "---- Received onBackInvoked ----")
                onBackPress()
            }
            // use fully qualified class name otherwise crashes on < 33 api devices
            getOnBackInvokedDispatcher().registerOnBackInvokedCallback(0, onBackPressedCallback as android.window.OnBackInvokedCallback)
        }

        ActivityUtils.allowShowFromLockscreen(TAG, this)
        savedInstanceState?.let {
            restore = it.getBoolean("restore", false)
        }
        errorCallback =
            TRErrorCallback { trError -> LogUtils.internal(TAG, "error: $trError") }
        TapResearch.orchestrator?.showContentForPlacementErrorCallback?.let {
            this.errorCallback = it
        }
        if (TapResearch.orchestrator == null) {
            finish()
            return
        }
        val toolbar = Toolbar(this)
        setupConfig()
        try {
            val webView = initWebView() // can fail when system is heavily resource constrained
            setupToolbar(toolbar, webView)
            configureWebViewCallback()
            configureWebView(webView)

            val layout = LinearLayout(this).apply {
                orientation = LinearLayout.VERTICAL // Setting the orientation to vertical
                layoutParams = LinearLayout.LayoutParams(
                    LinearLayout.LayoutParams.MATCH_PARENT,
                    LinearLayout.LayoutParams.MATCH_PARENT,
                )
                adjustEdgeToEdgeMargins(this)
                addView(toolbar)
                addView(webView)
            }

            setContentView(layout)
            if (!isLargeScreenDevice(this)) {
                webView.post({ requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_PORTRAIT })
            }
            TapResearch.webviewTypeOpen = WebViewTypeOpen.SURVEY_WALL

        } catch (t: Throwable) {
            RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TRWebViewActivity.onCreate", "Exception occurred: ", t)
            finish()
        }
    }

    override fun onResume() {
        super.onResume()
        if (TapResearch.orchestrator == null || TapResearch.orchestrator?.surveyWebView == null) {
            LogUtils.e(TAG, "onResume() orchestrator null.  exit survey wall.")
            finish()
        }
    }

    private fun setupConfig() {
        intent.extras?.getString("configuration")?.let { configString ->
            config = json.decodeFromString(configString)
        }
    }

    companion object {

        internal fun setUrlQueryParameters(url: String, params: HashMap<String, String>): String {
            return params.entries.fold(url) { newUrl, (param, value) ->
                UrlQuerySanitizer(newUrl).getValue(param).let { currentValue ->
                    when (currentValue) {
                        null -> {
                            // LogUtils.i("TRWebViewActivity", "Param NOT found [$param]: inserting $value")
                            Uri.parse(newUrl).buildUpon().appendQueryParameter(
                                param,
                                value,
                            ).build().toString()
                        }

                        value -> {
                            newUrl
                        }

                        else -> {
                            // LogUtils.i("TRWebViewActivity", "Param found [$param]: replaced with $value")
                            newUrl.replace(currentValue, value)
                        }
                    }
                }
            }
        }

        @OptIn(InternalSerializationApi::class)
        internal fun getUrlWithSecurityToken(url: String): String? {
            val currentPlacementTag = TapResearch.currentPlacement?.tag ?: return null
            val secHash = TapResearch.securityHashes?.get(currentPlacementTag) ?: return null
            val newUrl = setUrlQueryParameters(
                url,
                hashMapOf(
                    "sec_token" to secHash.token!!,
                    "sec_timestamp" to secHash.timestamp!!,
                ),
            )
            return if (newUrl != url) newUrl else null
        }

        internal fun didLoadUrlWithToken(url: String?, view: WebView): Boolean? {
            try {
                val urlString = url.toString()
                val paths = TapResearch.securityUrlPaths
                if (paths.isNullOrEmpty() || !paths.any { urlString.contains(it) }) return null

                getUrlWithSecurityToken(urlString)?.let {
                    // LogUtils.e("TRWebViewActivity", "Loading URL with new tokens: $it")
                    view.loadUrl(it)
                    return true
                }

                return null
            } catch (e: Exception) {
                val errorDetail = SdkEventDetail(
                    message = "[didLoadUrlWithToken] ${e.message}",
                    className = "TRWebViewActivity",
                )
                TapResearch.orchestrator?.captureSdkEvent(
                    errorDetail,
                    EventType.ERROR,
                    LogLevel.ERROR,
                    false,
                )
            }
            return null
        }
    }


    private fun pageReloadCallback(webView: WebView) {
        didLoadUrlWithToken(webView.url, webView)?.let {
            return
        }
        webView.reload()
    }

    private fun setupToolbar(toolbar: Toolbar, webView: WebView) {
        // Configure the toolbar: set visibility, title, background, and back button
        config.navigationBar?.let { navigationBar ->
            with(toolbar) {
                title = navigationBar.title
                background = ColorDrawable(Color.parseColor(navigationBar.backgroundColor))
                navigationIcon = ContextCompat.getDrawable(
                    this@TRWebViewActivity,
                    R.drawable.baseline_arrow_back_black_18,
                )

                setNavigationOnClickListener {
                    val orchestrator = TapResearch.orchestrator
                    if (backButtonState == TRWebViewState.ShowCloseButton) {
                        orchestrator?.evaluateJavascript(TapConstants.CLOSE_SURVEY_WALL)
                    } else {
                        showAbandonDialog(webView)
                    }
                }

                val reloadDrawable = ContextCompat.getDrawable(
                    context,
                    R.drawable.ic_reload_black_24dp,
                )
                val reloadButton = ImageButton(context).apply {
                    setImageDrawable(reloadDrawable)
                    setBackgroundColor(Color.TRANSPARENT)
                    layoutParams = Toolbar.LayoutParams(
                        Toolbar.LayoutParams.WRAP_CONTENT,
                        Toolbar.LayoutParams.WRAP_CONTENT,
                        Gravity.END,
                    ).apply {
                        marginEnd = 16.dpToPx(context) // Add margin as needed, convert dp to px
                    }
                    setOnClickListener {
                        pageReloadCallback(webView)
                    }
                }

                val rotateDrawable = ContextCompat.getDrawable(
                    context,
                    R.drawable.mobile_devices_rotate,
                )
                rotateDrawable?.setTint(Color.BLACK)
                val rotateButton = ImageButton(context).apply {
                    setImageDrawable(rotateDrawable)
                    setBackgroundColor(Color.TRANSPARENT)
                    layoutParams = Toolbar.LayoutParams(
                        Toolbar.LayoutParams.WRAP_CONTENT,
                        Toolbar.LayoutParams.WRAP_CONTENT,
                        Gravity.END,
                    ).apply {
                        marginEnd = 28.dpToPx(context) // Add margin as needed, convert dp to px
                    }
                    setOnClickListener {
                        rotateScreen()
                    }
                }
                rotateButton.id = R.id.tr_rotate_button_id

                // Add the TextView to the Toolbar
                addView(reloadButton)
                addView(rotateButton)
                if (isLargeScreenDevice(this@TRWebViewActivity)) {
                    rotateButton.visibility = View.GONE
                }
            }
        }
    }

    private fun rotateScreen() {
        if (isLargeScreenDevice(this)) return
        requestedOrientation =
            if (resources.configuration.orientation == Configuration.ORIENTATION_PORTRAIT) {
                ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE
            } else {
                ActivityInfo.SCREEN_ORIENTATION_PORTRAIT
            }
    }

    // Extension function to convert dp to pixels
    private fun Int.dpToPx(context: Context): Int {
        return TypedValue.applyDimension(
            TypedValue.COMPLEX_UNIT_DIP,
            this.toFloat(),
            context.resources.displayMetrics,
        ).toInt()
    }

    private fun handleOfferNavigation(offerUrl: String, context: Context) {
        // LogUtils.d("TRWebViewActivity", "Handling offer navigation: $offerUrl")
        try {
            val intent = Intent(Intent.ACTION_VIEW, Uri.parse(offerUrl))
            context.startActivity(intent)
        } catch (e: Exception) {
            val errorDetail = SdkEventDetail(
                message = "[handleOfferNavigation] ${e.message}",
                className = "TRWebViewActivity",
            )
            TapResearch.orchestrator?.captureSdkEvent(
                errorDetail,
                EventType.ERROR,
                LogLevel.ERROR,
                false,
            )
        }
    }

    @SuppressLint("SetJavaScriptEnabled")
    private fun initWebView(): WebView {
        detachWebView()
        if (restore) {
            restore = false
            return TapResearch.orchestrator?.surveyWebView!!
        } else {
            TapResearch.orchestrator?.surveyWebView = WebView(this).apply {
                settings.javaScriptEnabled = true
                settings.domStorageEnabled = true
                this.addJavascriptInterface(
                    TRSurveyInterface(
                        offerNavigationCallback = { offerUrl ->
                            handleOfferNavigation(
                                offerUrl,
                                this.context,
                            )
                        },
                    ),
                    "Android",
                )
                val webViewMessage = TapResearch.childWebViewMessage()
                val transport = webViewMessage?.obj as WebView.WebViewTransport?

                transport?.webView = this

                if (webViewMessage?.target != null) {
                    webViewMessage.sendToTarget()
                } else {
                    val errorDetail = SdkEventDetail(
                        message = "WebViewTransport - Target Was Null",
                        className = "TRWebViewActivity",
                    )
                    TapResearch.orchestrator?.captureSdkEvent(
                        errorDetail,
                        EventType.ERROR,
                        LogLevel.ERROR,
                        true,
                    )
                }

                webViewClient = object : WebViewClient() {

                    override fun onRenderProcessGone(view: WebView?, detail: RenderProcessGoneDetail?): Boolean {
                        try {
                            (view?.parent as ViewGroup).removeView(view)
                        } catch (ignored: Throwable) { }
                        TapResearch.orchestrator?.surveyWebView?.webViewClient = WebViewClient()
                        TapResearch.orchestrator?.surveyWebView?.destroy()
                        TapResearch.orchestrator?.surveyWebView = null
                        RemoteEventLogger.postEvent(
                            REMOTE_LOG_LEVEL,
                            "TRWebViewActivity.initWebView",
                            "content webview received onRenderProcessGone from system to reclaim resources."
                        )
                        finish()
                        return true
                    }

                    // This method is called when a new URL is about to be loaded in the current WebView.
                    override fun shouldOverrideUrlLoading(view: WebView, request: WebResourceRequest): Boolean {
                        // Get the URL from the request.
                        val url = request.url.toString()

                        // LogUtils.d("WebViewClient", "OVERRIDE URL LOADING: $url")

                        // Router URL redirect handled with security token
                        didLoadUrlWithToken(url, view)?.let { return true }

                        // If the WebView's current URL is null, load the requested URL.
                        if (view.url == null) {
                            view.loadUrl(url)
                            // Return true because the URL loading has been handled by our code.
                            return true
                        }

                        // Return false to indicate that the URL loading has not been handled by our code.
                        // The WebView will load the URL itself.
                        return false
                    }

                    override fun onPageFinished(view: WebView?, url: String?) {
                        super.onPageFinished(view, url)
                        sendNavigationMessage(view, null)
                    }

                    override fun onReceivedError(
                        view: WebView?,
                        request: WebResourceRequest?,
                        error: WebResourceError?,
                    ) {
                        super.onReceivedError(view, request, error)
                        sendNavigationMessage(view, error)
                    }

                    private fun sendNavigationMessage(view: WebView?, error: WebResourceError?) {
                        val code = error?.errorCode
                        val description = error?.description
                        val url = view?.url
                        val message = "handleSurveyWallRedirect('$url', '$description', '$code')"
                        TapResearch.orchestrator?.evaluateJavascript(message)
                    }
                }
            }
        }
        return TapResearch.orchestrator?.surveyWebView!!
    }

    private fun configureWebView(webView: WebView) {
        webView.apply {
            isVerticalScrollBarEnabled = true
            isHorizontalScrollBarEnabled = true

            val canScroll = config.webview?.scrolling == 1
            isVerticalScrollBarEnabled = canScroll

            layoutParams = RelativeLayout.LayoutParams(
                withDensity(config.webview?.frame?.width)
                    ?: LinearLayout.LayoutParams.MATCH_PARENT,
                withDensity(config.webview?.frame?.height)
                    ?: LinearLayout.LayoutParams.MATCH_PARENT,
            ).apply {
                leftMargin = withDensity(config.webview?.frame?.x) ?: 0
                topMargin = withDensity(config.webview?.frame?.y) ?: 0
            }

            config.webview?.layer?.let {
                background = GradientDrawable().apply {
                    it.cornerRadius?.let { cornerRadius = it }
                }
            }
        }
    }

    private fun configureWebViewCallback() {
        backButtonState = TRWebViewState.ShowCloseButton

        TapResearch.setOrchestratorWebViewCallback { webViewState ->
            when (webViewState) {
                is TRWebViewState.CloseWebView -> {
                    this.finish()
                }

                TRWebViewState.ShowAbandonButton -> {
                    backButtonState = TRWebViewState.ShowAbandonButton
                }

                TRWebViewState.ShowCloseButton -> {
                    backButtonState = TRWebViewState.ShowCloseButton
                }

                is TRWebViewState.UpdateWebViewDimensions -> {}
            }
        }
    }

    @Deprecated("Deprecated in Java")
    override fun onBackPressed() {
        LogUtils.internal(TAG, "---- Received onBackPressed ----")
        onBackPress()
    }

    private fun showAbandonDialog(webView: WebView) {
        val alertDialogBuilder = AlertDialog.Builder(this, android.R.style.Theme_Material_Light_Dialog)
        alertDialogBuilder.apply {
            setTitle(config.navigationBar?.abandonNavItem?.title)
            setMessage(config.navigationBar?.abandonNavItem?.abandonModal?.message)
            setPositiveButton(config.navigationBar?.abandonNavItem?.abandonModal?.actionTitle) { _, _ ->
                TapResearch.orchestrator?.evaluateJavascript(
                    message = TapConstants.ABANDON_SURVEY,
                    callback = { response ->
                        response?.let {
                            val abandonUrl = it.replace("\"", "")
                            if (URLUtil.isValidUrl(abandonUrl)) {
                                webView.loadUrl(abandonUrl)
                            }
                        }
                    },
                )
            }
            setNegativeButton(config.navigationBar?.abandonNavItem?.abandonModal?.dismissTitle) { dialog, _ ->
                dialog.dismiss()
            }
        }
        val dialog = alertDialogBuilder.create()
        dialog.show()
        config.navigationBar?.let { navigationBar ->
            dialog.window?.setBackgroundDrawable(ColorDrawable(Color.parseColor(navigationBar.backgroundColor)))
        }
    }

    override fun onDestroy() {
        LogUtils.internal(TAG, "onDestroy()")
        super.onDestroy()

        if (backButtonState == TRWebViewState.ShowCloseButton) {
            TapResearch.orchestrator?.evaluateJavascript(TapConstants.CLOSE_SURVEY_WALL)
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            LogUtils.internal(TAG, "---- Unregister onBackInvoked callback ----")
            // use fully qualified class name otherwise crashes on < 33 api devices
            try {
                if (onBackPressedCallback != null) {
                    getOnBackInvokedDispatcher().unregisterOnBackInvokedCallback(
                        onBackPressedCallback as android.window.OnBackInvokedCallback
                    )
                }
            } catch (ignored: Throwable){}
        }

        detachWebView()
        if (!restore) {
            destroyWebView()
        }
        TapResearch.setOrchestratorWebViewCallback(null)
        TapResearch.orchestrator?.webViewMessage = null
        TapResearch.webviewTypeOpen = WebViewTypeOpen.NONE
    }

    private fun withDensity(num: Float?): Int? {
        val density = Resources.getSystem().displayMetrics.density
        return num?.let { (it * density).toInt() }
    }

    private fun destroyWebView() {
        TapResearch.orchestrator?.surveyWebView?.let { webView ->
            try {
                webView.destroy()
                LogUtils.internal(TAG, "destroyWebView(). destroyed webView")
            } catch (e: Exception) {
                LogUtils.internal(TAG, "destroyWebView(). did not destroy webView: $e")
            }
        }
        TapResearch.orchestrator?.surveyWebView = null
    }

    private fun detachWebView() {
        TapResearch.orchestrator?.surveyWebView?.let { webView ->
            try {
                val viewGroupParent = webView.parent as ViewGroup
                viewGroupParent.removeView(webView)
                LogUtils.internal(TAG, "detachWebView(). removed global webView from parent")
            } catch (e: Exception) {
                LogUtils.internal(TAG, "detachWebView(). global webView does not have parent (ok)")
            }
        }
    }

    override fun onSaveInstanceState(outState: Bundle) {
        super.onSaveInstanceState(outState)
        LogUtils.internal(TAG, "onSaveInstanceState()")
        outState.putBoolean("restore", true)
        restore = true
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        try {
            val rotateButton = findViewById<View>(R.id.tr_rotate_button_id)
            rotateButton.visibility = if (isLargeScreenDevice(this)) View.GONE else View.VISIBLE
        }catch (_: Throwable){}
    }
}
