package com.tapresearch.tapsdk.callback

import android.webkit.JavascriptInterface
import com.tapresearch.tapsdk.SdkEventDetail
import com.tapresearch.tapsdk.SecurityHash
import com.tapresearch.tapsdk.TapResearch
import com.tapresearch.tapsdk.models.QuickQuestion
import com.tapresearch.tapsdk.models.TRError
import com.tapresearch.tapsdk.models.TRInitPayload
import com.tapresearch.tapsdk.models.TRPlacement
import com.tapresearch.tapsdk.models.TRReward
import com.tapresearch.tapsdk.models.TRSurveyRefreshPayload
import com.tapresearch.tapsdk.models.configuration.DialogDimensions
import com.tapresearch.tapsdk.state.EventType
import com.tapresearch.tapsdk.state.LogLevel
import com.tapresearch.tapsdk.state.TRWebViewState
import com.tapresearch.tapsdk.storage.PlacementStorage
import com.tapresearch.tapsdk.storage.SdkFeatureToggleStorage
import com.tapresearch.tapsdk.utils.LogUtils
import com.tapresearch.tapsdk.utils.RemoteEventLogger
import com.tapresearch.tapsdk.utils.TapConstants
import com.tapresearch.tapsdk.utils.TapConstants.REMOTE_LOG_LEVEL
import com.tapresearch.tapsdk.utils.TapConstants.RESTART_TIMER
import com.tapresearch.tapsdk.utils.TimerUtil
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.serialization.InternalSerializationApi
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.SerializationException
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonArray
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.jsonObject
import org.json.JSONObject
import kotlin.math.absoluteValue

@InternalSerializationApi @Serializable
internal data class SecurityUrlPathsPayload(
    @SerialName("url_paths") val urlPaths: Array<String>? = null,
)

internal class TRJSInterface(
    internal var rewardCallback: TRRewardCallback? = null,
    private var presentationCallback: (configuration: String) -> Unit,
    private var contentCallback: TRContentCallback?,
    private var webViewCallback: ((webViewState: TRWebViewState) -> Unit)?,
    private var onInitializedCallback: (payload: TRInitPayload) -> Unit,
    private var onErrorCallback: TRErrorCallback?,
    internal var qqDataCallback: TRQQDataCallback?,
) {

    private val json = Json { ignoreUnknownKeys = true }
    internal var surveysRefreshedListener: TRSurveysRefreshedListener? = null

    internal fun resetRewardCallback(
        rewardCallback: TRRewardCallback? = null,
    ) {
        this.rewardCallback = rewardCallback
    }

    internal fun resetQQDataCallback(
        qqDataCallback: TRQQDataCallback? = null,
    ) {
        this.qqDataCallback = qqDataCallback
    }

    internal fun resetCallbacks(
        rewardCallback: TRRewardCallback? = null,
        presentationCallback: (configuration: String) -> Unit,
        contentCallback: TRContentCallback?,
        webViewCallback: ((webViewState: TRWebViewState) -> Unit)?,
        onInitializedCallback: (payload: TRInitPayload) -> Unit,
        onErrorCallback: TRErrorCallback?,
        qqDataCallback: TRQQDataCallback?,
    ) {
        this.rewardCallback = rewardCallback
        this.presentationCallback = presentationCallback
        this.contentCallback = contentCallback
        this.webViewCallback = webViewCallback
        this.onInitializedCallback = onInitializedCallback
        this.onErrorCallback = onErrorCallback
        this.qqDataCallback = qqDataCallback
    }

    @JavascriptInterface
    fun onAppInitialized(sdkInitPayload: String) {
        LogUtils.internal("JavaScriptInterface", "onAppInitialized: $sdkInitPayload")

        TapResearch.onOrcaCanRespond()

        val trInitPayload: TRInitPayload? = json.decodeFromString<TRInitPayload>(sdkInitPayload)

        trInitPayload?.let {
            CoroutineScope(Dispatchers.Main).launch {
                try {
                    onInitializedCallback(it)
                } catch (throwable: Throwable) {
                    RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TRJSInterface.onAppInitialized", "Publisher implementation failed: ", throwable)
                }
            }
            it.logLevel?.let { logLevel ->
                LogUtils.currentLogLevel = LogLevel.values()[logLevel.absoluteValue]
            }
        }
    }

    @JavascriptInterface
    fun onReadyToInitialize(message: String) {
        LogUtils.internal("JavaScriptInterface", "onReadyToInitialize: $message")
        TapResearch.onOrcaCanRespond()
        TapResearch.runManualInitialization()
    }

    @OptIn(InternalSerializationApi::class)
    @JavascriptInterface
    fun onSetSecurityUrlPaths(message: String) {
        LogUtils.internal("JavaScriptInterface", "onSetSecurityUrlPaths: $message")
        try {
            val payload: SecurityUrlPathsPayload? = json.decodeFromString(message)

            LogUtils.internal("JavaScriptInterface", "Url payload decoded: $payload")
            if (payload?.urlPaths?.isNotEmpty() == true) {
                TapResearch.securityUrlPaths = payload.urlPaths
            }
        } catch (throwable: Throwable) {
            RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TRJSInterface.onSetSecurityUrlPaths", "failed. message: $message", throwable)
        }
    }

    @OptIn(InternalSerializationApi::class)
    @JavascriptInterface
    fun onStoreSecurityHash(message: String) {
        LogUtils.internal("JavaScriptInterface", "onStoreSecurityHash: $message")
        val hashPayload: SecurityHash? = json.decodeFromString(message)
        if (hashPayload != null) {
            val placementTag = hashPayload.placementTag ?: return
            TapResearch.securityHashes?.put(placementTag, hashPayload)
        }
    }

    /****************** Placements ******************/
    @JavascriptInterface
    fun onPlacementReady(placementString: String) {
        // Log the received placementString
        LogUtils.internal("JavaScriptInterface", "onPlacementReady: $placementString")
        TapResearch.onOrcaCanRespond()
        // Add the placementString to PlacementStorage
        with(PlacementStorage) {
            try {
                add(decode(placementString))
            } catch (error: Throwable) {
                captureSdkErrorEvent("onPlacementReady failed. placement = $placementString  error = ${error.message}")
            }
        }
    }

    @JavascriptInterface
    fun onPlacementUnavailable(placementString: String) {
        // handle the onPlacementReady event here
        LogUtils.internal("JavaScriptInterface", "onPlacementUnavailable: $placementString")
        TapResearch.onOrcaCanRespond()

        with(PlacementStorage) {
            try {
                decode(placementString).let {
                    when (it.error) {
                        null -> remove(it)
                        else -> add(it)
                    }
                }
            } catch (error: Throwable) {
                captureSdkErrorEvent("onPlacementUnavailable failed. placement = $placementString  error = ${error.message}")
            }
        }
    }

    /****************** Rewards ******************/

    @JavascriptInterface
    fun onTapResearchDidReceiveRewards(rewardsJson: String) {
        LogUtils.internal("JavaScriptInterface", "onTapResearchDidReceiveRewards: $rewardsJson")
        TapResearch.onOrcaCanRespond()
        val jsonObject: JsonObject = Json.parseToJsonElement(rewardsJson).jsonObject
        val rewards = jsonObject["rewards"] as JsonArray

        val rewardArray: MutableList<TRReward> = ArrayList()
        for (i in 0 until rewards.size) {
            val rewardJsonString = rewards[i].toString()
            val reward = json.decodeFromString<TRReward>(rewardJsonString)
            rewardArray.add(reward)
        }
        CoroutineScope(Dispatchers.Main).launch {
            try {
                rewardCallback?.onTapResearchDidReceiveRewards(rewardArray)
            } catch (throwable: Throwable) {
                RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TRJSInterface.onTapResearchDidReceiveRewards", "Publisher implementation failed: ", throwable)
            }
        }
    }

    /****************** Quick Questions ******************/

    @JavascriptInterface
    fun onTapResearchDataReceived(dataJson: String) {
        LogUtils.internal("JavaScriptInterface", "onTapResearchDataReceived: $dataJson")
        val jsonObject = JSONObject(dataJson)
        if (jsonObject.optString("type", "").equals("quick_questions") && jsonObject.has("payload")) {
            val data = json.decodeFromString<QuickQuestion>(dataJson)
            CoroutineScope(Dispatchers.Main).launch {
                try {
                    qqDataCallback?.onQuickQuestionDataReceived(data.payload)
                } catch (throwable: Throwable) {
                    RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TRJSInterface.onTapResearchDataReceived", "Publisher implementation failed: ", throwable)
                }
            }
        }
    }

    /****************** Presentation ******************/

    @JavascriptInterface
    fun presentContentWithConfiguration(configurationJson: String) {
        LogUtils.internal("JavaScriptInterface", "presentContentWithConfiguration: $configurationJson")

        // This info needs to be passed through an intent. We'll decode inside the TRWebViewActivity
        CoroutineScope(Dispatchers.Main).launch {
            try {
                presentationCallback(configurationJson)
            } catch (throwable: Throwable) {
                RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TRJSInterface.presentContentWithConfiguration", "callback failed: ", throwable)
            }
        }
    }

    /****************** Content ******************/

    @JavascriptInterface
    fun onTapResearchContentShownForPlacement(placementJson: String) {
        LogUtils.internal("JavaScriptInterface", "onTapResearchContentShownForPlacement: $placementJson")

        val placement = json.decodeFromString<TRPlacement>(placementJson)
        CoroutineScope(Dispatchers.Main).launch {
            try {
                contentCallback?.onTapResearchContentShown(placement.tag ?: "")
            } catch (throwable: Throwable) {
                RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TRJSInterface.onTapResearchContentShownForPlacement", "callback failed: ", throwable)
            }
        }
    }

    @JavascriptInterface
    fun onTapResearchContentDismissedForPlacement(placementJson: String) {
        LogUtils.internal(
                "JavaScriptInterface",
                "onTapResearchContentDismissedForPlacement: $placementJson",
        )

        val placement = json.decodeFromString<TRPlacement>(placementJson)
        CoroutineScope(Dispatchers.Main).launch {
            try {
                contentCallback?.onTapResearchContentDismissed(placement.tag ?: "")
            } catch (throwable: Throwable) {
                RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TRJSInterface.onTapResearchContentDismissedForPlacement", "callback failed: ", throwable)
            }
        }
    }

    @JavascriptInterface
    fun onTapResearchDidError(error: String) {
        val trError = json.decodeFromString<TRError>(error)
        LogUtils.internal("JavaScriptInterface", "onTapResearchDidError: $trError")
        if (TapResearch.orchestrator?.isRestarting == true) {
            TimerUtil.stopTimer(RESTART_TIMER)
            TapResearch.orchestrator?.onOrcaRestartFailed()
            RemoteEventLogger.postEvent(
                REMOTE_LOG_LEVEL,
                "TRJSInterface.onTapResearchDidError",
                "$trError",
                null,
                TapConstants.LOG_CAT_ORCA_REVIVE,
                TapConstants.LOG_DET_ORCA_REVIVE_FAIL)
        }
        CoroutineScope(Dispatchers.Main).launch {
            try {
                onErrorCallback?.onTapResearchDidError(trError)
            } catch (throwable: Throwable) {
                RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TRJSInterface.onTapResearchDidError", "callback failed: ", throwable)
            }
        }
    }

    /****************** Webview ******************/

    @JavascriptInterface
    fun showAbandonButton(message: String) {
        LogUtils.internal("JavaScriptInterface", "showAbandonButton")
        CoroutineScope(Dispatchers.Main).launch {
            try {
                webViewCallback?.invoke(TRWebViewState.ShowAbandonButton)
            } catch (throwable: Throwable) {
                RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TRJSInterface.showAbandonButton", "callback failed: ", throwable)
            }
        }
    }

    @JavascriptInterface
    fun showCloseButton(message: String) {
        LogUtils.internal("JavaScriptInterface", "showCloseButton")
        CoroutineScope(Dispatchers.Main).launch {
            try {
                webViewCallback?.invoke(TRWebViewState.ShowCloseButton)
            } catch (throwable: Throwable) {
                RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TRJSInterface.showCloseButton", "callback failed: ", throwable)
            }
        }
    }

    @JavascriptInterface
    fun onSetSdkToggles(message: String) {
        TapResearch.onOrcaCanRespond()
        LogUtils.internal("JavaScriptInterface", "onSetSdkToggles $message")
        with(SdkFeatureToggleStorage) {
            try {
                decode(message).forEach { t -> update(t) }
            } catch (error: Throwable) {
                captureSdkErrorEvent("onSetSdkToggles failed. message = $message  error = ${error.message}")
            }
        }
    }

    @JavascriptInterface
    fun closeWebView(message: String) {
        LogUtils.internal("JavaScriptInterface", "closeWebView")
        CoroutineScope(Dispatchers.Main).launch {
            try {
                webViewCallback?.invoke(TRWebViewState.CloseWebView)
            } catch (throwable: Throwable) {
                RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TRJSInterface.closeWebView", "callback failed: ", throwable)
            }
        }
    }

    @JavascriptInterface
    fun updateWebViewDimensions(dimensions: String) {
        LogUtils.internal("JavaScriptInterface", "updateWebViewDimensions: $dimensions")
        val dimensionsObj: DialogDimensions? = try {
            // Deserialize the JSON string into a Dimensions object
            json.decodeFromString<DialogDimensions>(dimensions)
        } catch (e: SerializationException) {
            LogUtils.e("JavaScriptInterface", "Failed to parse dimensions: $e")
            null
        }

        dimensionsObj?.let {
            // Pass the height value to TRWebViewState.UpdateWebViewDimensions
            CoroutineScope(Dispatchers.Main).launch {
                try {
                    webViewCallback?.invoke(TRWebViewState.UpdateWebViewDimensions(it.height))
                } catch (throwable: Throwable) {
                    RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TRJSInterface.updateWebViewDimensions", "callback failed: ", throwable)
                }
            }
        }
    }

    @JavascriptInterface
    fun onSurveysRefreshedForPlacement(message: String) {
        LogUtils.internal("JavaScriptInterface", "onSurveysRefreshedForPlacement: $message")
        val hashPayload: TRSurveyRefreshPayload? = json.decodeFromString(message)

        if (hashPayload != null) {
            val placementTag = hashPayload.placementTag ?: return
            surveysRefreshedListener?.let { listener ->
                CoroutineScope(Dispatchers.Main).launch {
                    try {
                        listener.onSurveysRefreshedForPlacement(placementTag)
                    } catch (throwable: Throwable) {
                        RemoteEventLogger.postEvent(
                            REMOTE_LOG_LEVEL,
                            "TRJSInterface.onSurveysRefreshedForPlacement",
                            "callback failed ($placementTag): ",
                            throwable
                        )
                    }
                }
            }

        }
    }

    private fun captureSdkErrorEvent(message: String) {
        val errorDetail = SdkEventDetail(
            message = message,
            className = "TRJSInterface",
        )
        TapResearch.orchestrator?.captureSdkEvent(
            errorDetail,
            EventType.LOG,
            LogLevel.ERROR,
            false,
        )
    }
}
