package com.tapresearch.tapsdk

import android.annotation.SuppressLint
import android.app.Activity
import android.app.Application
import android.content.ComponentCallbacks
import android.content.ComponentCallbacks2
import android.content.res.Configuration
import android.os.Bundle
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import com.tapresearch.tapsdk.utils.LogUtils

internal class TRAppLifecycleObserver() :
    ComponentCallbacks,
    ComponentCallbacks2,
    DefaultLifecycleObserver,
    Application.ActivityLifecycleCallbacks {
    // A Tag to label logs generated by this class
    private val TAG = "TRLog_LCO"

    // Called when the owner LifecycleOwner moves to the onResume state.
    // This callback tells the orchestrator that the app is active.
    override fun onResume(owner: LifecycleOwner) {
        LogUtils.internal(TAG, "onResume")
        TapResearch.orchestrator?.evaluateJavascript("appDidBecomeActive()")
        TapResearch.appInForeground = true
    }

    // Called when the owner LifecycleOwner moves to the onPause state.
    // This callback tells the orchestrator that the app entered the background.
    override fun onPause(owner: LifecycleOwner) {
        LogUtils.internal(TAG, "onPause")
        TapResearch.orchestrator?.evaluateJavascript("appDidEnterBackground()")
        TapResearch.appInForeground = false
    }

    // Called when the owner LifecycleOwner moves to the onStop state.
    // This callback also tells the orchestrator that the app entered the background.
    override fun onStop(owner: LifecycleOwner) {
        LogUtils.internal(TAG, "onStop")
    }

    // Called when the owner LifecycleOwner moves to the onDestroy state.
    // This callback tells the orchestrator that the app entered the background.
    override fun onDestroy(owner: LifecycleOwner) {
        LogUtils.internal(TAG, "onDestroy")
    }

    // Called by the system when the device configuration changes while your component is running.
    override fun onConfigurationChanged(config: Configuration) {
        // no op
        LogUtils.internal(TAG, "onConfigurationChanged")
        TapResearch.trDialogNonActivity?.onConfigurationChanged(config)
    }

    // This is called when the overall system is running low on memory
    // Deprecated since API 14, keep empty if not needed
    override fun onLowMemory() {
        // no op
    }

    // Called when the operating system has determined that it is a good time for a process
    // to trim unneeded memory from its process.
    @SuppressLint("SwitchIntDef")
    override fun onTrimMemory(level: Int) {
        when (level) {
            // The system is running critically low on memory notify orca.
            ComponentCallbacks2.TRIM_MEMORY_RUNNING_CRITICAL -> {
                LogUtils.internal(TAG, "Critical Memory Pressure")
                TapResearch.orchestrator?.evaluateJavascript("outOfMemoryWarning()")
            }
        }
    }

    // Called when an activity is starting.
    override fun onActivityCreated(
        activity: Activity,
        savedInstanceState: Bundle?,
    ) {
        LogUtils.internal(TAG, "onActivityCreated: ${activity::class.java.simpleName}")
        TapResearch.orchestrator?.storeTag()
        TapResearch.trDialogNonActivity?.onCreate(activity, savedInstanceState)
    }

    override fun onActivityStarted(activity: Activity) {
        LogUtils.internal(TAG, "onActivityStarted: ${activity::class.java.simpleName}")
    }

    override fun onActivityResumed(activity: Activity) {
        LogUtils.internal(TAG, "onActivityResumed: ${activity::class.java.simpleName}")
        TapResearch.trDialogNonActivity?.onResume(activity)
    }

    override fun onActivityPaused(activity: Activity) {
        LogUtils.internal(TAG, "onActivityPaused: ${activity::class.java.simpleName}")
    }

    override fun onActivityStopped(activity: Activity) {
        LogUtils.internal(TAG, "onActivityStopped: ${activity::class.java.simpleName}")
    }

    // Called as part of the activity lifecycle when an activity is going into the background, but has not (yet) been saved.
    // In here we save the state of whether the dialog was showing or not
    override fun onActivitySaveInstanceState(
        activity: Activity,
        outState: Bundle,
    ) {
        LogUtils.internal(TAG, "onActivitySaveInstanceState: ${activity::class.java.simpleName}")
        TapResearch.trDialogNonActivity?.onSaveInstanceState(activity, outState)
    }

    override fun onActivityDestroyed(activity: Activity) {
        LogUtils.internal(TAG, "onActivityDestroyed: ${activity::class.java.simpleName}")
        TapResearch.trDialogNonActivity?.onDestroy(activity)
    }

    override fun onCreate(owner: LifecycleOwner) {
        // not implemented
    }
    override fun onStart(owner: LifecycleOwner) {
        // not implemented
    }

    override fun onActivityPreCreated(activity: Activity,
                                      savedInstanceState: Bundle?) {
        // not implemented
    }

    /**
     * Called as the last step of the Activity being created. This is always called after
     * [Activity.onCreate].
     */
    override fun onActivityPostCreated(activity: Activity,
                                       savedInstanceState: Bundle?) {
        // not implemented
    }

    /**
     * Called as the first step of the Activity being started. This is always called before
     * [Activity.onStart].
     */
    override fun onActivityPreStarted(activity: Activity) {
        // not implemented
    }

    /**
     * Called as the last step of the Activity being started. This is always called after
     * [Activity.onStart].
     */
    override fun onActivityPostStarted(activity: Activity) {
        // not implemented
    }

    /**
     * Called as the first step of the Activity being resumed. This is always called before
     * [Activity.onResume].
     */
    override fun onActivityPreResumed(activity: Activity) {
        // not implemented
    }

    /**
     * Called as the last step of the Activity being resumed. This is always called after
     * [Activity.onResume] and [Activity.onPostResume].
     */
    override fun onActivityPostResumed(activity: Activity) {
        // not implemented
    }

    /**
     * Called as the first step of the Activity being paused. This is always called before
     * [Activity.onPause].
     */
    override fun onActivityPrePaused(activity: Activity) {
        // not implemented
    }

    /**
     * Called as the last step of the Activity being paused. This is always called after
     * [Activity.onPause].
     */
    override fun onActivityPostPaused(activity: Activity) {
        // not implemented
    }

    /**
     * Called as the first step of the Activity being stopped. This is always called before
     * [Activity.onStop].
     */
    override fun onActivityPreStopped(activity: Activity) {
        // not implemented
    }

    /**
     * Called as the last step of the Activity being stopped. This is always called after
     * [Activity.onStop].
     */
    override fun onActivityPostStopped(activity: Activity) {
        // not implemented
    }

    /**
     * Called as the first step of the Activity saving its instance state. This is always
     * called before [Activity.onSaveInstanceState].
     */
    override fun onActivityPreSaveInstanceState(activity: Activity,
                                                outState: Bundle) {
        // not implemented
    }

    /**
     * Called as the last step of the Activity saving its instance state. This is always
     * called after[Activity.onSaveInstanceState].
     */
    override fun onActivityPostSaveInstanceState(activity: Activity,
                                                 outState: Bundle) {
        // not implemented
    }

    /**
     * Called as the first step of the Activity being destroyed. This is always called before
     * [Activity.onDestroy].
     */
    override fun onActivityPreDestroyed(activity: Activity) {
        // not implemented
    }


    /**
     * Called as the last step of the Activity being destroyed. This is always called after
     * [Activity.onDestroy].
     */
    override fun onActivityPostDestroyed(activity: Activity) {
        // not implemented
    }
}
