package com.tapresearch.tapsdk.webview

import android.animation.ValueAnimator
import android.annotation.SuppressLint
import android.app.Activity
import android.app.AlertDialog
import android.app.Dialog
import android.content.DialogInterface
import android.content.Intent
import android.content.res.Configuration
import android.content.res.Resources
import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.graphics.drawable.GradientDrawable
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.view.Gravity
import android.view.KeyEvent
import android.view.MotionEvent
import android.view.ViewGroup
import android.view.WindowManager
import android.webkit.RenderProcessGoneDetail
import android.webkit.URLUtil
import android.webkit.WebResourceError
import android.webkit.WebResourceRequest
import android.webkit.WebView
import android.webkit.WebViewClient
import com.tapresearch.tapresearchkotlinsdk.R
import com.tapresearch.tapsdk.TapResearch
import com.tapresearch.tapsdk.TapResearch.webviewTypeOpen
import com.tapresearch.tapsdk.WebViewTypeOpen
import com.tapresearch.tapsdk.callback.TRErrorCallback
import com.tapresearch.tapsdk.models.TRError
import com.tapresearch.tapsdk.models.configuration.TRConfiguration
import com.tapresearch.tapsdk.state.TRWebViewState
import com.tapresearch.tapsdk.utils.DeviceUtils.isPortrait
import com.tapresearch.tapsdk.utils.LogUtils
import com.tapresearch.tapsdk.utils.RemoteEventLogger
import com.tapresearch.tapsdk.utils.TapConstants
import com.tapresearch.tapsdk.utils.TapConstants.REMOTE_LOG_LEVEL
import com.tapresearch.tapsdk.utils.TapErrorCodes
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.lang.ref.WeakReference

internal class TRDialogNonActivity(private val onFinished: () -> Unit) {

    companion object {
        private const val TAG = "TRLog_NDA"
    }

    private lateinit var dialog: Dialog
    private var dialogPreviousHeight = -1
    private lateinit var backButtonState: TRWebViewState
    private var portraitConfiguration: TRConfiguration? = null
    private var landscapeConfiguration: TRConfiguration? = null
    private var origPortraitY = 0f
    private var origLandscapeY = 0f
    private var errorCallback = TRErrorCallback {}
    private var doAnimation = true
    private var restore = false
    private var message = ""
    private var activity: WeakReference<Activity>? = null

    private fun activity(): Activity? {
        activity?.get()?.let {
            return it
        }
        return null
    }

    fun startFauxActivity(activity: Activity, portraitConfiguration: TRConfiguration?, landscapeConfiguration: TRConfiguration?) {
        this.portraitConfiguration = portraitConfiguration
        this.landscapeConfiguration = landscapeConfiguration
        this.origPortraitY = portraitConfiguration?.webview?.frame?.y?:0f
        this.origLandscapeY = landscapeConfiguration?.webview?.frame?.y?:0f
        this.activity = WeakReference(activity)
        onCreate(activity, null)
    }

    // faux override
    @SuppressLint("ClickableViewAccessibility")
    fun onCreate(activity: Activity, savedInstanceState: Bundle?) {
        try {
            activity()?.let { thisActivity ->
                if (thisActivity != activity) {
                    if (savedInstanceState == null || !savedInstanceState.containsKey("restore")) {
                        return // must be a new activity different from dialog activity
                    }
                }
            }
            this.activity = WeakReference(activity)
            savedInstanceState?.let {
                restore = it.getBoolean("restore", false)
            }
            LogUtils.internal(
                TAG,
                "onCreate sdkState: ${TapResearch.sdkState()} " + " orcaWebView: ${TapResearch.orchestrator?.orcaWebView} " + " webViewMessage: ${TapResearch.orchestrator?.webViewMessage?.what} " + " ${TapResearch.orchestrator?.webViewMessage?.arg1} " + " ${TapResearch.orchestrator?.webViewMessage?.arg2} " + " ${TapResearch.orchestrator?.webViewMessage?.obj} ",
            )
            errorCallback = TRErrorCallback { trError -> LogUtils.internal(TAG, "error: $trError") }
            TapResearch.orchestrator?.showContentForPlacementErrorCallback?.let {
                this.errorCallback = it
            }
            openDialog()
            webviewTypeOpen = WebViewTypeOpen.INTERSTITIAL
        } catch (throwable: Throwable) {
            RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TRDialogNonActivity.onCreate", "Exception occurred: ", throwable)
            cancel()
        }
    }

    private fun openDialog() {
        activity()?.let {
            createDialog(errorCallback, if (isPortrait()) portraitConfiguration else landscapeConfiguration)
        }
    }

    private fun createDialog(
        errorCallback: TRErrorCallback,
        dialogConfiguration: TRConfiguration?,
    ) {
        activity()?.let {

            try {
                dialog = Dialog(it)

                // Initialize WebView
                val webView = initWebView()

                // Configure WebView callback and handle back button presses
                configureWebViewCallback(dialog)
                setDialogOnBackPressed(dialog = dialog, trConfig = dialogConfiguration)

                // Add the layout to the dialog and display the dialog
                dialog.addContentView(webView, configureWebView(webView, dialogConfiguration))
            }  catch (t: Throwable) {
                RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TRDialogNonActivity.createDialog", "Exception occurred: ", t)
                cancel()
                return
            }

            // Set dialog dimensions
            setDialogDimensions(
                dialog,
                dialogConfiguration,
                errorCallback = errorCallback,
                isResize = false,
            )

            // Set dialog behavior on outside click
            if (dialogConfiguration?.view?.allowsTouchPassthrough == 1) {
                dialog.setCanceledOnTouchOutside(true)
            } else {
                dialog.setCanceledOnTouchOutside(false)
            }
            dialog.setOnCancelListener {
                TapResearch.orchestrator?.evaluateJavascript(TapConstants.CLOSE_DIALOG)
                portraitConfiguration = null
                landscapeConfiguration = null
            }

            if (dialogConfiguration == null) {
                onDialogConfigError(errorCallback)
                return
            }

            dialog.window?.apply {
                setWindowAnimations(R.style.DialogAnimation)
                val dimAmount = dialogConfiguration.view?.alpha ?: 0f
                setDimAmount(dimAmount)
            }
        }
    }

    private fun onDialogConfigError(
        errorCallback: TRErrorCallback,
    ) {
        TapResearch.orchestrator?.evaluateJavascript(TapConstants.CLOSE_DIALOG)

        errorCallback.onTapResearchDidError(
            TRError(
                code = TapErrorCodes.MISSING_ATTRIBUTE.code,
                description = TapErrorCodes.MISSING_ATTRIBUTE.errorMessage(),
            ),
        )
        cancel()
    }

    private fun setDialogDimensions(
        dialog: Dialog,
        webViewConfig: TRConfiguration?,
        errorCallback: TRErrorCallback,
        isResize: Boolean,
    ) {
        if (webViewConfig == null) {
            if (isResize) onDialogConfigError(errorCallback)
            return
        }
        val frame = webViewConfig.webview?.frame
        val window = dialog.window

        // Set dialog dimensions
        val layoutParams = WindowManager.LayoutParams().apply {
            copyFrom(window?.attributes)

            this.windowAnimations = R.style.QuickQuestionsAnimation

            gravity = if (dialogPreviousHeight != -1 && isResize) {
                Gravity.BOTTOM or Gravity.START // We need this for x and y to work properly
            } else {
                Gravity.TOP or Gravity.START // We need this for x and y to work properly
            }

            activity()?.let { activity ->
                frame?.let {
                    TRDialogUtils.adjustFramePosition(activity, it, origPortraitY, origPortraitY)
                }
            }

            if (frame?.width != null) {
                width = withDensityForCoordinates(frame.width)!!
            } else {
                errorCallback.onTapResearchDidError(
                    TRError(
                        code = TapErrorCodes.MISSING_ATTRIBUTE.code,
                        description = TapErrorCodes.MISSING_ATTRIBUTE.errorMessage(),
                    ),
                )
                return
            }

            if (frame.height != null) {
                height = if (dialogPreviousHeight != -1 && isResize) {
                    dialogPreviousHeight
                } else {
                    withDensityForCoordinates(frame.height)!!
                }
            } else {
                errorCallback.onTapResearchDidError(
                    TRError(
                        code = TapErrorCodes.MISSING_ATTRIBUTE.code,
                        description = TapErrorCodes.MISSING_ATTRIBUTE.errorMessage(),
                    ),
                )
                return
            }

            x = withDensityForCoordinates(frame.x) ?: 0
            y = withDensityForCoordinates(frame.y) ?: 0
        }

        window?.apply {
            attributes = layoutParams
            setBackgroundDrawable(getRoundedCornerDrawable())
        }
        dialog.show()
        LogUtils.internal(TAG, "dialog.show()")
    }

    private fun getRoundedCornerDrawable(): GradientDrawable {
        return GradientDrawable().apply {
            shape = GradientDrawable.RECTANGLE
            cornerRadii = FloatArray(8) { 20f } // All corners have the same radius
            setColor(Color.TRANSPARENT)
        }
    }

    private fun setDialogOnBackPressed(dialog: Dialog, trConfig: TRConfiguration?) {
        dialog.setOnKeyListener(
            DialogInterface.OnKeyListener { _, keyCode, event ->
                if (keyCode == KeyEvent.KEYCODE_BACK && event.action == KeyEvent.ACTION_UP) {
                    if (backButtonState == TRWebViewState.ShowCloseButton) {
                        cancel()
                        TapResearch.orchestrator?.evaluateJavascript(TapConstants.CLOSE_DIALOG)
                    } else {
                        showAbandonDialog(trConfig)
                    }
                    return@OnKeyListener true
                }
                false
            },
        )
    }

    @SuppressLint("SetJavaScriptEnabled")
    private fun initWebView(): WebView {
        detachWebView()
        if (restore) {
            restore = false
            return TapResearch.orchestrator?.dialogWebView!!
        } else {
            val webViewMessage = TapResearch.childWebViewMessage()
            if (webViewMessage == null || webViewMessage.obj == null) {
                RemoteEventLogger.postEvent(
                    REMOTE_LOG_LEVEL,
                    "TRDialogNonActivity.initWebView",
                    "webViewMessage or its object is null",
                )
                cancel()
            }
            activity()?.let {
                TapResearch.orchestrator?.dialogWebView = WebView(it).apply {
                    settings.javaScriptEnabled = true
                    settings.domStorageEnabled = true
                    setBackgroundColor(Color.TRANSPARENT) // eliminate white block button
                    webViewMessage?.obj?.let { someObj ->
                        try {
                            val transport = someObj as WebView.WebViewTransport
                            transport.webView = this
                            webViewMessage.sendToTarget()
                        } catch (throwable: Throwable) {
                            RemoteEventLogger.postEvent(
                                REMOTE_LOG_LEVEL,
                                "TRDialogNonActivity.initWebView",
                                "webView transport failed",
                                throwable = throwable,
                            )
                            cancel()
                        }
                    }
                    webViewClient = object : WebViewClient() {

                        override fun onRenderProcessGone(view: WebView?, detail: RenderProcessGoneDetail?): Boolean {
                            try {
                                (view?.parent as ViewGroup).removeView(view)
                            } catch (ignored: Throwable) { }
                            TapResearch.orchestrator?.dialogWebView?.webViewClient = WebViewClient()
                            TapResearch.orchestrator?.dialogWebView?.destroy()
                            TapResearch.orchestrator?.dialogWebView = null
                            RemoteEventLogger.postEvent(
                                REMOTE_LOG_LEVEL,
                                "TRDialogNonActivity.initWebView",
                                "content webview received onRenderProcessGone from system to reclaim resources."
                            )
                            cancel()
                            return true
                        }

                        override fun shouldOverrideUrlLoading(
                            view: WebView,
                            request: WebResourceRequest,
                        ): Boolean {
                            val legalUrl = "https://www.tapresearch.com/legal/"

                            val requestUrl = request.url.toString()

                            if (requestUrl.startsWith(legalUrl)) {
                                startActivityForUrl(request.url)
                                return true
                            }

                            // Check if the URL is valid and load it into WebView
                            if (URLUtil.isValidUrl(requestUrl)) {
                                view.loadUrl(requestUrl)
                            }
                            return false
                        }

                        /**
                         * Start a new activity for a given URL
                         */
                        private fun startActivityForUrl(url: Uri) {
                            activity()?.let {
                                val intent = Intent(it, TRSimpleWebView::class.java)
                                intent.data = url
                                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
                                it.startActivity(intent)
                            }
                        }

                        override fun onPageFinished(view: WebView?, url: String?) {
                            super.onPageFinished(view, url)
                            sendNavigationMessage(view, null)
                        }

                        override fun onReceivedError(
                            view: WebView?,
                            request: WebResourceRequest?,
                            error: WebResourceError?,
                        ) {
                            super.onReceivedError(view, request, error)
                            sendNavigationMessage(view, error)
                        }

                        private fun sendNavigationMessage(
                            view: WebView?,
                            error: WebResourceError?,
                        ) {
                            val code = error?.errorCode
                            val description = error?.description
                            val url = view?.url
                            message = "handleSurveyWallRedirect('$url', '$description', '$code')"
                            TapResearch.orchestrator?.evaluateJavascript(message)
                        }
                    }
                }
            }
            return TapResearch.orchestrator?.dialogWebView!!
        }
    }

    private fun configureWebViewCallback(dialog: Dialog) {
        backButtonState = TRWebViewState.ShowCloseButton

        TapResearch.setOrchestratorWebViewCallback { webViewState ->
            when (webViewState) {
                is TRWebViewState.CloseWebView -> {
                    cancel()
                }

                TRWebViewState.ShowAbandonButton -> {
                    backButtonState = TRWebViewState.ShowAbandonButton
                }

                TRWebViewState.ShowCloseButton -> {
                    backButtonState = TRWebViewState.ShowCloseButton
                }

                is TRWebViewState.UpdateWebViewDimensions -> {
                    CoroutineScope(Dispatchers.Main).launch {
                        val newHeight = withDensityForCoordinates(webViewState.height)
                        dialogPreviousHeight = newHeight!!
                        val window = dialog.window
                        val params = window?.attributes

                        // Set the gravity to bottom.
                        params?.gravity = Gravity.BOTTOM or Gravity.START
                        window?.attributes = params

                        if (doAnimation) {
                            doAnimation = false
                            // Initialize a ValueAnimator
                            val animator = ValueAnimator.ofInt(params?.height ?: 0, newHeight)
                            animator.duration = 20 // Duration of the animation
                            animator.addUpdateListener { animation ->
                                params?.height = animation.animatedValue as Int
                                window?.attributes = params
                            }

                            when (params?.height) {
                                newHeight -> {
                                    params.windowAnimations = R.style.QuickQuestionsAnimation
                                }
                            }
                            animator.start()
                        } else {
                            newHeight.apply {
                                params?.height = this
                            }
                            window?.attributes = params
                        }
                    }
                }
            }
        }
    }

    @SuppressLint("ClickableViewAccessibility")
    private fun configureWebView(
        webView: WebView,
        config: TRConfiguration?,
    ): ViewGroup.LayoutParams? {
        webView.apply {
            isVerticalScrollBarEnabled = true
            isHorizontalScrollBarEnabled = true

            val canScroll = config?.webview?.scrolling == 1
            isVerticalScrollBarEnabled = canScroll

            layoutParams = ViewGroup.LayoutParams(-1, -1)

            // provide way to quit modal dialog if orca dies mid-session
            setOnTouchListener { v, event ->
                if (event.action == MotionEvent.ACTION_DOWN) {
                    TapResearch.orchestrator?.apply {
                        checkOrca { isOk, reason ->
                            if (!isOk) {
                                v.setOnTouchListener(null)
                                RemoteEventLogger.postEvent(
                                    REMOTE_LOG_LEVEL,
                                    "TRDialogNonActivity.onTouchListener",
                                    "$reason. closing dialog. message: $message",
                                    null,
                                    TapConstants.LOG_CAT_ORCA_REVIVE,
                                    TapConstants.LOG_DET_ORCA_REVIVE_ATTEMPT
                                )
                                restart() // orca
                                cancel() // modal dialog
                            }
                        }
                    }
                }
                false
            }
        }
        return webView.layoutParams
    }

    private fun withDensityForCoordinates(num: Float?): Int? {
        val density = Resources.getSystem().displayMetrics.density
        return num?.let { (it * density).toInt() }
    }

    private fun showAbandonDialog(config: TRConfiguration?) {
        activity()?.let {
            val alertDialogBuilder = AlertDialog.Builder(it, android.R.style.Theme_Material_Light_Dialog)
            alertDialogBuilder.apply {
                setTitle(config?.navigationBar?.abandonNavItem?.title)
                setMessage(config?.navigationBar?.abandonNavItem?.abandonModal?.message)
                setPositiveButton(config?.navigationBar?.abandonNavItem?.abandonModal?.actionTitle) { _, _ ->
                    TapResearch.orchestrator?.evaluateJavascript(TapConstants.ABANDON_SURVEY)
                }
                setNegativeButton(config?.navigationBar?.abandonNavItem?.abandonModal?.dismissTitle) { _, _ ->
                    cancel()
                }
            }
            val dialog = alertDialogBuilder.create()
            dialog.show()
            config?.navigationBar?.let { navigationBar ->
                try {
                    dialog.window?.setBackgroundDrawable(ColorDrawable(Color.parseColor(navigationBar.backgroundColor)))
                } catch (e: Exception) {
                    LogUtils.e(TAG, "showAbandonDialog() could not set background drawable: $e")
                }
            }
        }
    }

    // faux override
    fun onConfigurationChanged(newConfig: Configuration) {
        if (::dialog.isInitialized && dialog.isShowing) {
            dialog.dismiss() // fixes pixel 6 api-33 where cant scroll after portrait to landscape
        }
        LogUtils.internal(TAG,"onConfigurationChanged new -> ${newConfig.orientation}",)
        activity()?.let {
            if (isPortrait()) {
                setDialogDimensions(dialog,portraitConfiguration,errorCallback,true)
            } else {
                fix33()
                setDialogDimensions(dialog,landscapeConfiguration,errorCallback,true)
            }
        }
    }

    /**
     * Fix33
     *
     * Fixes pixel 6 api-33 where cant scroll after portrait to landscape.
     * Many things were tried.  i.e. webview.requestLayout, enableScrolling, etc
     *
     * Does not affect non-33 api levels, but don't apply fix if not necessary.
     *
     */
    private fun fix33() {
        if (Build.VERSION.SDK_INT == 33 ) {
            if (::dialog.isInitialized && dialog.isShowing) {
                dialog.dismiss() // tried dialog hide, webview enableScrolling, requestLayout, etc
            }
        }
    }

    // faux override
    fun onResume(activity: Activity) {
        activity()?.let {
            if (it == activity) {
                LogUtils.internal(TAG, "onResume()")
                if (TapResearch.orchestrator == null || TapResearch.orchestrator?.dialogWebView == null) {
                    LogUtils.e(TAG, "onResume() orchestrator null.  exit interstitial dialog.")
                    finish()
                }
            }
        }
    }

    // faux override
    fun onPause(activity: Activity) {
        activity()?.let {
            if (it == activity) {
                LogUtils.internal(TAG, "onPause()")
            }
        }
    }

    // faux override
    fun onDestroy(activity: Activity) {
        activity()?.let {
            if (it == activity) {
                LogUtils.internal(TAG, "onDestroy() restore state: $restore")
                if (::dialog.isInitialized && dialog.isShowing) {
                    LogUtils.internal(TAG, "onDestroy dialog was showing, cancel")
                    if (restore) {
                        dialog.dismiss() // user left but wants to return
                    } else {
                        dialog.cancel() // user explicitly quit
                    }
                }
                detachWebView()
                if (!restore) {
                    destroyWebView()
                    onFinished()
                }
            }
        }
        webviewTypeOpen = WebViewTypeOpen.NONE
        TapResearch.trDialogNonActivity = null
    }

    private fun cancel() {
        if (::dialog.isInitialized && dialog.isShowing) {
            dialog.cancel()
            LogUtils.internal(TAG, "cancel dialog and invisible activity")
        }
        finish()
    }

    // faux override
    private fun finish() {
        CoroutineScope(Dispatchers.Main).launch {
            activity()?.let {
                onDestroy(it)
            }
        }
    }

    private fun destroyWebView() {
        TapResearch.orchestrator?.dialogWebView?.let { webView ->
            try {
                webView.destroy()
                LogUtils.internal(TAG, "destroyWebView(). destroyed webView")
            } catch (e: Exception) {
                LogUtils.internal(TAG, "destroyWebView(). did not destroy webView: $e")
            }
        }
        TapResearch.orchestrator?.dialogWebView = null
    }

    private fun detachWebView() {
        TapResearch.orchestrator?.dialogWebView?.let { webView ->
            try {
                val viewGroupParent = webView.parent as ViewGroup
                viewGroupParent.removeView(webView)
                LogUtils.internal(TAG, "detachWebView(). removed global webView from parent")
            } catch (e: Exception) {
                LogUtils.internal(TAG, "detachWebView(). global webView does not have parent (ok)")
            }
        }
    }

    // faux override
    fun onSaveInstanceState(activity: Activity, outState: Bundle) {
        activity()?.let {
            if (it == activity) {
                LogUtils.internal(TAG, "onSaveInstanceState()")
                outState.putBoolean("restore", true)
                restore = true
            }
        }
    }
}
