package com.tapresearch.tapsdk.utils

import com.tapresearch.tapresearchkotlinsdk.BuildConfig
import com.tapresearch.tapsdk.SdkEventDetail
import com.tapresearch.tapsdk.TapResearch
import com.tapresearch.tapsdk.state.EventType
import com.tapresearch.tapsdk.state.LogLevel
import com.tapresearch.tapsdk.utils.TapConstants.REMOTE_LOG_LEVEL
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

/**
 * Show content logger
 *
 * Tracks when TapResearch.showContentForPlacement() or TapResearch.canShowContentForPlacement()
 * fails so as to help determine possible publisher SDK misuse and/or missed opportunities.
 *
 * @constructor Create empty Show content logger
 */
internal object ShowContentLogger {

    private const val MAX_LOGS = 20
    private var timeout = 60000L // milliseconds
    private var timestamp = 0L
    private var logCount = 0

    internal fun interface TRShowContentLoggerCallback {
        /**
         * On log complete
         *
         * @param result:  0=did not log, 1=logged via orca, 2=logged natively
         */
        fun onLogComplete(result: Int)
    }

    internal fun setTimeout(timeout: Long) {
        if (BuildConfig.DEBUG) {
            this.timeout = timeout
        }
    }

    internal fun log(tag: String, message: String, listener: TRShowContentLoggerCallback? = null) {

        if (!canLog()) {
            listener?.onLogComplete(0)
            return
        }

        // try to use the orca captureSdkEvent (sdk must be initialized and orca must not be dead)
        // otherwise, send the log natively
        if (TapResearch.orcaCanRespond && TapResearch.orchestrator != null) {
            TapResearch.orchestrator?.apply {
                CoroutineScope(Dispatchers.Main).launch {
                    checkOrca { orcaIsOk, _ ->
                        if (orcaIsOk) {
                            val errorDetail = SdkEventDetail(
                                message = "$tag $message",
                                className = "ShowContentLogger",
                            )
                            captureSdkEvent(
                                errorDetail,
                                EventType.LOG,
                                LogLevel.ERROR,
                                false,
                            )
                            incrementLogCount()
                            listener?.onLogComplete(1)
                        } else {
                            logNatively(tag, message, listener) // since orca not ok
                        }
                    }
                }
            }
        } else {
            logNatively(tag, message, listener) // since sdk not ready
        }
    }

    /**
     * Increment log count and reset the counter, if necessary.
     *
     * If more than one minute has elapsed since the last timestamp,
     * reset both the timestamp and logCount back to zero.
     */
    private fun incrementLogCount() {
        if (timestamp == 0L) {
            timestamp = System.currentTimeMillis()
        } else if (System.currentTimeMillis() - timestamp > timeout) {
            timestamp = 0L
            logCount = 0
        }
        logCount++
    }

    private fun logNatively(tag: String, message: String, listener: TRShowContentLoggerCallback?) {
        RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, tag, message)
        incrementLogCount()
        listener?.onLogComplete(2)
    }

    /**
     * Throttle show content log events if too many are coming in.
     *
     * If more than MAX_LOGS_PER_MINUTE have occurred in less than 1 minute, then
     * return false (don't allow)
     *
     * @return Boolean - can log or not
     */
    private fun canLog():Boolean {
        return (System.currentTimeMillis() - timestamp < timeout && logCount >= MAX_LOGS).not()
    }
}
