package com.tapresearch.tapsdk.utils

import com.tapresearch.tapsdk.utils.LogUtils.Companion.internal
import java.io.BufferedOutputStream
import java.io.BufferedReader
import java.io.IOException
import java.io.InputStream
import java.io.InputStreamReader
import java.net.HttpURLConnection
import java.net.URL
import java.net.URLConnection
import java.net.URLEncoder
import java.nio.charset.StandardCharsets
import java.util.zip.GZIPInputStream

/**
 *
 */
internal class HttpHelper {
    private val url: URL
    private var timeout = 0
    private var extraHeaders: Map<String, String>? = null

    internal class HttpResponse {
        var response: String? = null
        var httpResponseCode = 0
        override fun toString(): String {
            return "httpResponseCode: $httpResponseCode  response: $response"
        }
    }

    /**
     * Constructs a new HttpHelper.
     */
    constructor(url: String?) {
        this.url = URL(url)
    }

    constructor(url: String?, extraHeaders: Map<String, String>?) {
        this.url = URL(url)
        this.extraHeaders = extraHeaders
    }

    /**
     * @param url     - the url
     * @param timeout - timeout seconds to connect
     * @throws MalformedURLException
     */
    constructor(url: String?, timeout: Int) {
        this.url = URL(url)
        this.timeout = timeout
    }

    @get:Throws(IOException::class)
    val string: String
        get() {
            var `is`: InputStream? = null
            var httpURLConnection: HttpURLConnection? = null
            return try {
                val urlConnection = openConnection()
                httpURLConnection = urlConnection as HttpURLConnection
                assignCommonAttributes(httpURLConnection, true, false)
                httpURLConnection.setRequestMethod(GET)
                setExtraHeaders(httpURLConnection)
                httpURLConnection.setRequestProperty("Accept-Encoding", "gzip")
                `is` = httpURLConnection.inputStream
                getString(if (isResponseGzip(urlConnection)) GZIPInputStream(`is`) else `is`)
            } catch (e: IOException) {
                httpURLConnection?.disconnect()
                throw e
            } finally {
                try {
                    `is`?.close()
                } catch (e: Exception) {
                    internal(TAG, "getString exception: $e")
                }
            }
        }

    @Throws(IOException::class)
    fun get(): HttpResponse {
        var `is`: InputStream? = null
        var httpURLConnection: HttpURLConnection? = null
        return try {
            val urlConnection = openConnection()
            httpURLConnection = urlConnection as HttpURLConnection
            assignCommonAttributes(httpURLConnection, true, false)
            httpURLConnection.setRequestMethod(GET)
            setExtraHeaders(httpURLConnection)
            httpURLConnection.setRequestProperty("Accept-Encoding", "gzip")
            `is` = httpURLConnection.inputStream
            val response =
                    getString(if (isResponseGzip(urlConnection)) GZIPInputStream(`is`) else `is`)
            val httpResponse = HttpResponse()
            httpResponse.response = response
            httpResponse.httpResponseCode = urlConnection.getResponseCode()
            httpResponse
        } catch (e: IOException) {
            httpURLConnection?.disconnect()
            throw e
        } finally {
            try {
                `is`?.close()
            } catch (e: Exception) {
                internal(TAG, "getString() $e")
            }
        }
    }

    @Throws(IOException::class)
    private fun openConnection(): URLConnection {
        return url.openConnection()
    }

    @Throws(IOException::class)
    fun post(payload: String): HttpResponse {
        return postData(payload.toByteArray(), POST)
    }

    @Throws(IOException::class)
    fun put(payload: String): HttpResponse {
        return postData(payload.toByteArray(), PUT)
    }

    private fun assignCommonAttributes(
            urlConnection: URLConnection?,
            doInput: Boolean,
            doOutput: Boolean,
    ) {
        urlConnection?.apply {
            useCaches = false
            setDoInput(doInput)
            setDoOutput(doOutput)
            useCaches = false
            if (timeout > 0) {
                setConnectTimeout(1000 * timeout)
                setReadTimeout(1000 * timeout)
            } else {
                setConnectTimeout(30000)
                setReadTimeout(30000)
            }
        }
    }

    /**
     *
     * @param data
     * @param httpMethod
     * @return
     * @throws Exception
     */
    @Throws(IOException::class)
    private fun postData(data: ByteArray, httpMethod: String): HttpResponse {
        val urlConnection = openConnection()
        assignCommonAttributes(urlConnection, true, httpMethod != DELETE)
        (urlConnection as HttpURLConnection).setRequestMethod(httpMethod)
        setExtraHeaders(urlConnection)
        urlConnection.setRequestProperty("Content-Type", "application/json")
        urlConnection.setRequestProperty("Accept", "application/json")
        urlConnection.setRequestMethod(httpMethod)
        var wr: BufferedOutputStream? = null
        var rd: BufferedReader? = null
        return try {
            if (httpMethod != DELETE) {
                // Send post data request
                wr = BufferedOutputStream(urlConnection.getOutputStream())
                wr.write(data)
                wr.flush()
                wr.close()
                wr = null
            }

            // Get Response
            rd = if (isResponseGzip(urlConnection)) {
                BufferedReader(
                        InputStreamReader(
                                GZIPInputStream(urlConnection.getInputStream()),
                                StandardCharsets.UTF_8,
                        ),
                        BUFFERED_READER_SIZE,
                )
            } else {
                BufferedReader(
                        InputStreamReader(urlConnection.getInputStream(), UTF8),
                        BUFFERED_READER_SIZE,
                )
            }
            val response = StringBuilder()
            var line: String?
            while (rd.readLine().also { line = it } != null) {
                response.append(line)
            }
            rd.close()
            rd = null
            val httpResponse = HttpResponse()
            httpResponse.response = response.toString()
            httpResponse.httpResponseCode = urlConnection.getResponseCode()
            httpResponse
        } finally {
            try {
                rd?.close()
            } catch (ignored: IOException) {
            }
            try {
                wr?.close()
            } catch (ignored: IOException) {
            }
        }
    }

    @Throws(IOException::class)
    fun post(formParams: Map<String?, String?>?): HttpResponse {
        val params = StringBuilder()
        if (formParams != null) {
            for (key in formParams.keys) {
                params.append(
                        URLEncoder.encode(key, UTF8),
                ).append('=').append(
                        URLEncoder.encode(
                                formParams[key],
                                UTF8,
                        ),
                ).append('&')
            }
            params.deleteCharAt(params.length - 1)
        }
        return postData(params.toString().toByteArray(StandardCharsets.UTF_8), POST)
    }

    @Throws(IOException::class)
    fun delete(data: ByteArray): HttpResponse {
        return postData(data, DELETE)
    }

    @Throws(IOException::class)
    private fun getString(`is`: InputStream): String {
        val sb = StringBuilder(BUFFERED_READER_SIZE)
        val buf = ByteArray(BUFFERED_READER_SIZE)
        var read: Int
        while (`is`.read(buf).also { read = it } != -1) {
            sb.append(String(buf, 0, read, StandardCharsets.UTF_8))
        }
        try {
            `is`.close()
        } catch (ignored: Throwable) {
        }
        return sb.toString()
    }

    private fun isResponseGzip(urlConnection: URLConnection): Boolean {
        val responseHeaders = urlConnection.headerFields as Map<String, List<String>>
        for (key in responseHeaders.keys) {
            val values = responseHeaders[key]!!
            for (value in values) {
                internal(TAG, "response header: $key / $value")
                if (value.contains("gzip")) {
                    internal(TAG, "Response contains gzip: $value")
                    return true
                }
            }
        }
        return false
    }

    private fun setExtraHeaders(connection: URLConnection?) {
        extraHeaders?.apply {
            for (key in keys) {
                connection?.let { conn ->
                    conn.setRequestProperty(key, this[key])
                    internal(TAG, "set header: $key / ${this[key]}")
                }
            }
        }
    }

    companion object {
        private const val TAG = "HttpHelper"
        private const val POST = "POST"
        private const val DELETE = "DELETE"
        private const val PUT = "PUT"
        private const val GET = "GET"
        private const val UTF8 = "UTF-8"
        private const val BUFFERED_READER_SIZE = 8192
    }
}
