package com.tapresearch.tapsdk.storage

import android.annotation.SuppressLint
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.res.Resources
import android.os.BatteryManager
import android.os.Build
import android.provider.Settings
import android.util.Base64
import android.webkit.WebSettings
import com.google.android.gms.ads.identifier.AdvertisingIdClient
import com.tapresearch.tapresearchkotlinsdk.BuildConfig
import com.tapresearch.tapsdk.TapResearch
import com.tapresearch.tapsdk.utils.ActivityUtils
import com.tapresearch.tapsdk.utils.AppIdUtil
import com.tapresearch.tapsdk.utils.AppVersionHelper
import com.tapresearch.tapsdk.utils.ConnectionUtils
import com.tapresearch.tapsdk.utils.DeviceUtils.getDisplayDimensions
import com.tapresearch.tapsdk.utils.DeviceUtils.isLargeScreenDevice
import com.tapresearch.tapsdk.utils.DeviceUtils.isPortrait
import com.tapresearch.tapsdk.utils.LogUtils
import com.tapresearch.tapsdk.utils.PlatformUtils
import com.tapresearch.tapsdk.utils.RemoteEventLogger
import com.tapresearch.tapsdk.utils.StorageUtil
import com.tapresearch.tapsdk.utils.TapConstants.REMOTE_LOG_LEVEL
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.net.URLDecoder
import java.net.URLEncoder
import java.util.Locale


internal object ParameterStorage {

    private const val TAG = "ParameterStorage"

    // determining is emulator
    private const val GOLDFISH = "goldfish"
    private const val RANCHU = "ranchu"
    private const val SDK = "sdk"

    private const val PARAMS_PREFS = "tr_orca_params"

    private val urlParams = HashMap<String, String>()
    private val headers = HashMap<String, String>()
    internal var populated = false

    private val headersMap = mutableMapOf(
        "aid" to "X-Android-Id",
        "api_token" to "X-Api-Token",
        "app_version_name" to "X-App-Version-Name",
        "app_version_number" to "X-App-Version-Number",
        "battery_level" to "X-Battery-Level",
        "carrier" to "X-Carrier",
        "connection_type" to "X-Connection-Type",
        "dev_platform" to "X-Dev-Platform",
        "dev_version" to "X-Dev-Version",
        "device_brand" to "X-Device-Brand",
        "device_identifier" to "X-Device-Identifier",
        "device_name" to "X-Device-Name",
        "device_type" to "X-Device-Type",
        "display_dimensions" to "X-Display-Dimensions",
        "first_install_time" to "X-First-Install-Time",
        "free_disk_storage" to "X-Free-Disk-Storage",
        "has_notch" to "X-Has-Notch",
        "is_battery_charging" to "X-Is-Battery-Charging",
        "is_emulator" to "X-Is-Emulator",
        "orientation" to "X-Orientation",
        "os_version" to "X-OS-Version",
        "pixel_size" to "X-Pixel-Size",
        "pkg_name" to "X-Pkg-Name",
        "screen_scale_string" to "X-Screen-Scale",
        "sdk_platform" to "X-SDK-Platform",
        "sdk_version" to "X-SDK-Version",
        "supported_abis" to "X-Supported-Abis",
        "target_sdk_version" to "X-Target-SDK-Version",
        "total_disk_capacity" to "X-Total-Disk-Capacity",
        "total_memory" to "X-Total-Memory",
        "used_memory" to "X-Used-Memory",
        "user_agent" to "X-User-Agent",
        "user_identifier" to "X-User-Identifier",
        "locale" to "X-User-Locale",
        "vendor_identifier" to "X-Vendor-Identifier",
    )

    internal fun urlParams(): String {
        val stringBuilder = StringBuilder(4096)
        try {
            for (key in urlParams.keys) {
                // These are already added to the URL in TROrchestrator.kt
                if (key == "api_token" || key == "user_identifier") continue

                stringBuilder.append("&").append(key).append("=").append(urlParams[key])
            }
        } catch (e: Throwable) {
            RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "ParameterStorage", "failed-7", e)
        }
        return stringBuilder.toString()
    }

    private fun setHeaders() {
        try {
            val nativeHeaders = urlParams.mapKeys { headersMap[it.key] ?: it.key }.toMutableMap()
            nativeHeaders["X-User-Agent"]?.let {
                nativeHeaders["X-User-Agent"] = URLDecoder.decode(it, "UTF-8")
            }
            LogUtils.internal(TAG, "Setting native headers: $nativeHeaders")
            headers.putAll(nativeHeaders)
        } catch (e: Throwable) {
            RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "ParameterStorage", "failed-0", e)
        }
    }

    /**
     * Fetches parameters from disk.  Call in background thread.
     *
     *
     * @param context
     */
    internal fun populateFromIO(context: Context) {
        try {
            if (populated) return
            val sharedPrefs = context.getSharedPreferences(PARAMS_PREFS, 0)
            if (sharedPrefs.contains("first_install_time")) {
                urlParams["first_install_time"] =
                    sharedPrefs.getLong("first_install_time", System.currentTimeMillis()).toString()
            } else {
                sharedPrefs.edit().putLong(
                    "first_install_time",
                    System.currentTimeMillis(),
                ).apply() //millis since jan 1 1970
                urlParams["first_install_time"] = System.currentTimeMillis().toString()
            }
            if (sharedPrefs.contains("has_notch")) {
                urlParams["has_notch"] = sharedPrefs.getString("has_notch", "0").toString()
            }
            urlParams["device_identifier"] = getAdvertisingId(context) ?: "na"
            AppIdUtil.getAppId(context)?.let { urlParams["vendor_identifier"] = it }
            checkStorage(context)
            populated = true
        } catch (e: Throwable) {
            RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "ParameterStorage", "failed-1", e)
        }
    }

    /**
     * Populate - fetches parameters that do not require disk access.  Can be called on the main thread.
     *
     * @param context
     */
    internal fun populateFromMain(context: Context) {
        try {
            // params that don't need context
            urlParams["device_name"] = Build.MODEL.replace(' ', '-')
            urlParams["device_brand"] = Build.MANUFACTURER.replace(' ', '-') //android only
            urlParams["is_emulator"] = if (isEmulator()) "1" else "0"
            urlParams["supported_abis"] = supportedAbis()
            urlParams["sdk_platform"] = "android" // always "android"
            urlParams["sdk_version"] = BuildConfig.TAP_SDK_VERSION
            urlParams["dev_platform"] = PlatformUtils.devPlatform()
            urlParams["dev_version"] = PlatformUtils.devVersion()
            urlParams["pixel_size"] = getPixelSize()
            urlParams["locale"] = Locale.getDefault().language
            urlParams["os_version"] = Build.VERSION.SDK_INT.toString()
            urlParams["screen_scale_string"] =
                Resources.getSystem().displayMetrics.density.toString()

            // params that require context
            urlParams["pkg_name"] = context.packageName
            urlParams["target_sdk_version"] = context.applicationInfo.targetSdkVersion.toString()
            urlParams["orientation"] = if (isPortrait()) "portrait" else "landscape"
            urlParams["device_type"] = if (isLargeScreenDevice(context)) "tablet" else "phone"
            AppVersionHelper.getAppVersion(context)?.let {
                urlParams["app_version_name"] = it.versionName
                urlParams["app_version_number"] = it.versionNumber.toString()
            }
            urlParams["user_agent"] = userAgent(context)

            // older params
            urlParams["connection_type"] = ConnectionUtils.getConnectionType(context)
            urlParams["carrier"] = ConnectionUtils.getMobileCarrierName(context)
            urlParams["aid"] = aid(context) ?: "na"
            checkWiredHeadsets(context) // works fine
            checkBattery(context)
            urlParams["display_dimensions"] = getDisplayDimensions(context)
        } catch (e: Throwable) {
            RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "ParameterStorage", "failed-8", e)
        }
        setHeaders()
    }

    /**
     * Refresh only the refreshable items.  Call in background thread.
     *
     */
    internal fun refreshHeaders(): Map<String, String> {
        val copy = HashMap<String, String>()
        TapResearch.application?.let { context ->
            try {
                populateFromIO(context)
                checkBattery(context)
                checkStorage(context)
                checkWiredHeadsets(context)
                urlParams["connection_type"] = ConnectionUtils.getConnectionType(context)
                urlParams["orientation"] = if (isPortrait()) "portrait" else "landscape"
                headers.clear()
                setHeaders()
                copy.putAll(headers)
            } catch (e: Throwable) {
                RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "ParameterStorage", "failed-2", e)
            }
        }
        return copy
    }

    private fun userAgent(context: Context): String {
        return try {
            URLEncoder.encode(WebSettings.getDefaultUserAgent(context), "UTF-8")
        } catch (_: Throwable) {
            ""
        }
    }

    private fun supportedAbis(): String {
        // use try/catch, was crashing during unit test
        return try {
            Build.SUPPORTED_ABIS.asList().toString().replace(" ", "")
        } catch (_: Throwable) {
            ""
        }
    }

    @SuppressLint("HardwareIds")
    private fun aid(context: Context): String? {
        try {
            val aid = Settings.Secure.getString(context.contentResolver, Settings.Secure.ANDROID_ID)
            return Base64.encodeToString(aid.toByteArray(), Base64.URL_SAFE)
        } catch (_: Throwable) {
            return null
        }
    }

    private fun getAdvertisingId(context: Context): String? {
        var advertisingId: String? = null
        try {
            val sharedPrefs = context.getSharedPreferences(PARAMS_PREFS, 0)
            if (sharedPrefs.contains("device_id")) {
                return sharedPrefs.getString("device_id", "na")
            }
            val info = AdvertisingIdClient.getAdvertisingIdInfo(context)
            if (!info.isLimitAdTrackingEnabled) {
                advertisingId = info.id
                advertisingId?.let {
                    sharedPrefs.edit().putString("device_id", it).apply()
                }
            }
        } catch (e: Exception) {
            RemoteEventLogger.postEvent(
                REMOTE_LOG_LEVEL,
                "ParameterStorage_8",
                "failed to get GAD",
                e,
            )
        }
        return advertisingId
    }

    private fun getPixelSize(): String {
        val height = Resources.getSystem().displayMetrics.heightPixels
        val width = Resources.getSystem().displayMetrics.widthPixels
        return "${width}x$height"
    }

    internal fun saveHasNotch(context: Context, hasNotch: Boolean) {
        val hasNotchValue = if (hasNotch) "1" else "0"
        try {
            urlParams["has_notch"] = hasNotchValue
        } catch (_: Throwable) {
        }
        CoroutineScope(Dispatchers.IO).launch {
            try {
                context.getSharedPreferences(PARAMS_PREFS, 0).edit().putString(
                    "has_notch",
                    hasNotchValue,
                ).apply()
            } catch (_: Throwable) {
            }
        }
    }

    private fun checkStorage(context: Context) {
        try {
            urlParams["free_disk_storage"] = StorageUtil.freeDiskSpace().toString()
            urlParams["total_disk_capacity"] = StorageUtil.totalDiskSpace().toString()
            val memStats = ActivityUtils.usedMemory(context)
            memStats?.let {
                urlParams["total_memory"] = memStats[0]
                urlParams["used_memory"] = memStats[1]
            }
        } catch (e: Throwable) {
            RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "ParameterStorage", "failed-3", e)
        }
    }

    private fun checkBattery(context: Context) {
        try {
            val batteryManager = context.getSystemService(Context.BATTERY_SERVICE) as BatteryManager
            urlParams["is_battery_charging"] = if (batteryManager.isCharging) "1" else "0"
            val batLevel: Int =
                batteryManager.getIntProperty(BatteryManager.BATTERY_PROPERTY_CAPACITY)
            urlParams["battery_level"] = batLevel.toString()
        } catch (e: Throwable) {
            RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "ParameterStorage", "failed-4", e)
        }
    }

    private fun checkWiredHeadsets(context: Context) {
        val receiver = object : BroadcastReceiver() {
            override fun onReceive(context: Context?, intent: Intent) {
                try {
                    val action = intent.action
                    if (Intent.ACTION_HEADSET_PLUG == action) {
                        if (intent.getIntExtra("state", -1) == 0) {
                            // NOT plugged in
                            urlParams["is_headphones_connected"] = "0"
                        }
                        if (intent.getIntExtra("state", -1) == 1) {
                            // plugged in
                            urlParams["is_headphones_connected"] = "1"
                        }
                    }
                    context?.unregisterReceiver(this)
                } catch (_: Throwable) {
                }
            }
        }
        try {
            val receiverFilter = IntentFilter(Intent.ACTION_HEADSET_PLUG)
            context.registerReceiver(receiver, receiverFilter)
        } catch (_: Throwable) {
        }
    }

    private fun isEmulator(): Boolean {
        return if (Build.VERSION.SDK_INT >= 31) {
            (Build.PRODUCT.contains(SDK) || Build.HARDWARE.contains(GOLDFISH) || Build.HARDWARE.contains(
                RANCHU,
            ))
        } else {
            (Build.HARDWARE.contains(GOLDFISH) || Build.HARDWARE.contains(RANCHU))
        }
    }

    internal fun saveApiToken(apiToken: String) {
        try {
            headers["X-Api-Token"] = apiToken
            urlParams["api_token"] = apiToken
        } catch (e: Throwable) {
            RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "ParameterStorage", "failed-5", e)
        }
    }

    internal fun saveUserIdentifier(userIdentifier: String?) {
        try {
            headers["X-User-Identifier"] = userIdentifier ?: ""
            urlParams["user_identifier"] = userIdentifier ?: ""
        } catch (e: Throwable) {
            RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "ParameterStorage", "failed-6", e)
        }
    }
}
