package com.tapresearch.tapsdk

import android.annotation.SuppressLint
import android.app.Activity
import android.app.Application
import android.content.Context
import android.content.Intent
import android.os.Looper
import android.os.Message
import androidx.lifecycle.ProcessLifecycleOwner
import com.tapresearch.tapresearchkotlinsdk.BuildConfig
import com.tapresearch.tapsdk.TapResearch.getSurveysForPlacement
import com.tapresearch.tapsdk.callback.TRContentCallback
import com.tapresearch.tapsdk.callback.TRErrorCallback
import com.tapresearch.tapsdk.callback.TRQQDataCallback
import com.tapresearch.tapsdk.callback.TRRewardCallback
import com.tapresearch.tapsdk.callback.TRSdkReadyCallback
import com.tapresearch.tapsdk.callback.TRSurveysRefreshedListener
import com.tapresearch.tapsdk.models.OfferEventConfiguration
import com.tapresearch.tapsdk.models.TRError
import com.tapresearch.tapsdk.models.TRPlacement
import com.tapresearch.tapsdk.models.TRSurvey
import com.tapresearch.tapsdk.models.configuration.TRConfiguration
import com.tapresearch.tapsdk.receiver.ScreenONReceiver
import com.tapresearch.tapsdk.state.SdkState
import com.tapresearch.tapsdk.state.SdkStateHolder
import com.tapresearch.tapsdk.state.TRWebViewState
import com.tapresearch.tapsdk.storage.ParameterStorage
import com.tapresearch.tapsdk.storage.PlacementStorage
import com.tapresearch.tapsdk.utils.ConnectionUtils
import com.tapresearch.tapsdk.utils.LogUtils
import com.tapresearch.tapsdk.utils.RemoteEventLogger
import com.tapresearch.tapsdk.utils.ShowContentLogger
import com.tapresearch.tapsdk.utils.TRHelper.executeOrcaReadyTask
import com.tapresearch.tapsdk.utils.TRHelper.waitWhileInitializing
import com.tapresearch.tapsdk.utils.TapConstants.INITIALIZATION_TIMER
import com.tapresearch.tapsdk.utils.TapConstants.LOG_TAG
import com.tapresearch.tapsdk.utils.TapConstants.REMOTE_LOG_LEVEL
import com.tapresearch.tapsdk.utils.TapErrorCodes
import com.tapresearch.tapsdk.utils.TimerUtil
import com.tapresearch.tapsdk.webview.TRDialogActivity
import com.tapresearch.tapsdk.webview.TRDialogNonActivity
import com.tapresearch.tapsdk.webview.TROrchestrator
import com.tapresearch.tapsdk.webview.TRWebViewActivity
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.invoke
import kotlinx.coroutines.launch
import kotlinx.serialization.InternalSerializationApi
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.json.Json
import org.json.JSONObject
import java.lang.ref.WeakReference

/**
 * User attributes to send to TapResearch for targeting purposes when initializing the SDK.
 *
 * @property userAttributes Map of user attributes to send to TapResearch for targeting purposes.
 * @property clearPreviousAttributes Pass true to clear any previously sent user attributes; false otherwise
 * @constructor Create empty Tap init options
 *
 * @see TapResearch.initialize
 */
data class TapInitOptions(
    val userAttributes: HashMap<String, Any>? = null,
    val clearPreviousAttributes: Boolean? = false,
)

/**
 * Sdk event detail
 *
 * @property message
 * @property className
 * @constructor Create empty Sdk event detail
 */
internal data class SdkEventDetail(val message: String, val className: String)

/**
 * Web view type open
 *
 * @property parameterValue
 * @constructor Create empty Web view type open
 */
internal enum class WebViewTypeOpen(val parameterValue: String) {
    NONE("none"),
    INTERSTITIAL("interstitial"),
    SURVEY_WALL("survey_wall"),
}

/**
 * Security hash
 *
 * @property placementTag
 * @property token
 * @property timestamp
 * @constructor Create empty Security hash
 */
@InternalSerializationApi
@Serializable
internal data class SecurityHash(
    @SerialName("placement_tag") val placementTag: String? = null,

    @SerialName("sec_token") val token: String? = null,

    @SerialName("sec_timestamp") val timestamp: String? = null,
)

/**
 * TapResearch - All survey api are contained within this object.  Be sure to call initialize()
 * before any other method, such as showContentForPlacement().
 *
 *
 * @see initialize
 * @see showContentForPlacement
 */
object TapResearch {
    internal var userIdentifier: String = ""
    internal var initOptions: TapInitOptions? = null

    @SuppressLint("StaticFieldLeak")
    internal var orchestrator: TROrchestrator? = null

    internal var application: Context? = null
    internal var currentPlacement: TRPlacement? = null
    private var currentOfferConfiguration: TRConfiguration? = null
    private var configurationString: String = ""
    private var portraitConfiguration: TRConfiguration? = null
    private var landscapeConfiguration: TRConfiguration? = null
    private val lifecycleObserver: TRAppLifecycleObserver = TRAppLifecycleObserver()
    @OptIn(InternalSerializationApi::class)
    internal var securityHashes: MutableMap<String, SecurityHash>? = mutableMapOf()
    internal var orcaCanRespond = false
    internal val sdkStateHolder = SdkStateHolder()
    private val json = Json { ignoreUnknownKeys = true }
    private var activityWeakReference: WeakReference<Activity>? = null
    internal var trDialogNonActivity: TRDialogNonActivity? = null
    private var userErrorListener = TRErrorCallback{}
    private var userSdkReadyListener = TRSdkReadyCallback {}
    private var initializationResponseReceived = false
    internal var doTestSkipSdkInit = false // for testing
    internal var doTestSdkInitError = false // for testing
    internal var securityUrlPaths: Array<String>? = null
    internal var doTestSkipWall = false
    internal var wallWebViewIntent: Intent? = null
    internal var interstitialIntent: Intent? = null
    internal var appInForeground = false
    internal var webviewTypeOpen = WebViewTypeOpen.NONE

    /**
     * Sets the reward listener.
     *
     * @param rewardCallback Listens for reward events; such as when a survey is completed and
     * after the survey wall is closed.  Pass null to disable listening.
     *
     */
    fun setRewardCallback(
        rewardCallback: TRRewardCallback? = null,
    ) {
        executeOrcaReadyTask(
            jobName = "setRewardListener",
            orcaReadyTask = {
                orchestrator?.resetRewardCallback(rewardCallback)
                orchestrator?.notifyRewardCallbackAvailability(rewardCallback)
            }
        )
    }

    /**
     * Sets the quick question listener.
     *
     * @param quickQuestionCallback Listens for quick question events; such as when a quick
     * question has been answered along with question details.  Pass null to disable listening.
     *
     */
    fun setQuickQuestionCallback(
        quickQuestionCallback: TRQQDataCallback? = null,
    ) {
        executeOrcaReadyTask(
            jobName = "setQuickQuestionListener",
            orcaReadyTask = {
                orchestrator?.resetQQDataCallback(quickQuestionCallback)
                orchestrator?.notifyQQDataCallbackAvailability(quickQuestionCallback)
            }
        )
    }

    private val internalErrorListener = TRErrorCallback { error ->
        if (!initializationResponseReceived) {
            initializationResponseReceived = true
            // set to failed, otherwise sdk will be in a perpetual 'initializing' state
            // and won't be able to re-initialize since it will think it's already doing so
            sdkStateHolder.state = SdkState.initialization_failed
            orcaCanRespond = false
            TimerUtil.stopTimer(INITIALIZATION_TIMER)

            // 6-6-2025; no longer surface initialization timeout to publisher.
            if (error.code != TapErrorCodes.INITIALIZATION_TIMEOUT.code) {
                CoroutineScope(Dispatchers.Main).launch {
                    try {
                        userErrorListener.onTapResearchDidError(error) //always pass to user
                    } catch (throwable: Throwable) {
                        RemoteEventLogger.postEvent(
                            REMOTE_LOG_LEVEL,
                            "TapResearch.onTapResearchDidError",
                            "userErrorCallback failed. error: ${error.code} internal state: ${sdkStateHolder.state}",
                            throwable
                        )
                    }
                }
            }
        }
    }

    private val internalSdkReadyListener = TRSdkReadyCallback {
        if (!initializationResponseReceived) {
            initializationResponseReceived = true
            TimerUtil.stopTimer(INITIALIZATION_TIMER)
        }
        // 6-6-2025; do surface sdk ready regardless of any prior initialization errors
        CoroutineScope(Dispatchers.Main).launch {
            try {
                userSdkReadyListener.onTapResearchSdkReady()
            } catch (throwable: Throwable) {
                RemoteEventLogger.postEvent(
                    REMOTE_LOG_LEVEL,
                    "TapResearch.onTapResearchSdkReady",
                    "" +
                        "userSdkReadyCallback failed. internal state: ${sdkStateHolder.state}",
                    throwable
                )
            }
        }
    }

    /**
     * Initialize the TapResearch SDK.  Must be called before most TapResearch api.
     *
     * @param apiToken The API Token; provided by TapResearch team.
     * @param userIdentifier The unique user identifier; can be set again by calling setUserIdentifier.
     * @param context Activity or application context
     * @param rewardCallback Optional reward callback; can also call setRewardCallback
     * @param errorCallback Mandatory error callback; e.g. initialization errors.
     * @param sdkReadyCallback Mandatory SDK ready callback; let's you know the SDK is ready for use.
     * @param qqDataCallback Optional Quick Question data callback; also see setQuickQuestionCallback.
     * @param initOptions optional TapInitOptions.  Contains optional targeting parameters.
     *
     */
    fun initialize(
        apiToken: String,
        userIdentifier: String,
        context: Context,
        rewardCallback: TRRewardCallback? = null,
        errorCallback: TRErrorCallback,
        sdkReadyCallback: TRSdkReadyCallback,
        qqDataCallback: TRQQDataCallback? = null,
        initOptions: TapInitOptions? = null,
    ) {
        application = context.applicationContext
        if (Looper.myLooper() == Looper.getMainLooper()) {
            initializeOnMainThread(
                apiToken,
                userIdentifier,
                context.applicationContext,
                rewardCallback,
                errorCallback,
                sdkReadyCallback,
                qqDataCallback,
                initOptions,
            )
        } else {
            CoroutineScope(Dispatchers.Main).launch {
                initializeOnMainThread(
                    apiToken,
                    userIdentifier,
                    context.applicationContext,
                    rewardCallback,
                    errorCallback,
                    sdkReadyCallback,
                    qqDataCallback,
                    initOptions,
                )
            }
        }
    }

    /**
     * Initialize the TapResearch SDK.  Must be called before most TapResearch api.
     *
     * @param apiToken The API Token; provided by TapResearch team.
     * @param userIdentifier The unique user identifier; can be set again by calling setUserIdentifier.
     * @param activity Activity; application Context can also be used.
     * @param rewardCallback Optional reward callback; can also call SetRewardCallback
     * @param errorCallback Mandatory error callback; e.g. initialization errors.
     * @param sdkReadyCallback Mandatory SDK ready callback; let's you know the SDK is ready for use.
     * @param qqDataCallback Optional Quick Question data callback; also see setQuickQuestionCallback.
     * @param initOptions Optional TapInitOptions.  Contains optional targeting parameters.
     *
     */
    fun initialize(
        apiToken: String,
        userIdentifier: String,
        activity: Activity,
        rewardCallback: TRRewardCallback? = null,
        errorCallback: TRErrorCallback,
        sdkReadyCallback: TRSdkReadyCallback,
        qqDataCallback: TRQQDataCallback? = null,
        initOptions: TapInitOptions? = null,
    ) {
        initialize(
            apiToken = apiToken,
            userIdentifier = userIdentifier,
            context = activity,
            rewardCallback = rewardCallback,
            errorCallback = errorCallback,
            sdkReadyCallback = sdkReadyCallback,
            qqDataCallback = qqDataCallback,
            initOptions = initOptions
        )
    }

    private fun startInitializationTimer() {
        TimerUtil.startTimer(INITIALIZATION_TIMER, 20) {
            if (!initializationResponseReceived) {
                internalErrorListener.onTapResearchDidError(
                    TRError(
                        code = TapErrorCodes.INITIALIZATION_TIMEOUT.code,
                        description = TapErrorCodes.INITIALIZATION_TIMEOUT.errorMessage() + " -> [${sdkStateHolder.state}]",
                    )
                )
                RemoteEventLogger.postEvent(
                    REMOTE_LOG_LEVEL,
                    "TapResearch.startTimerTask",
                    "initialization timed out. internal state: ${sdkStateHolder.state}",
                )
            }
        }
    }

    private fun initializeOnMainThread(
        apiToken: String,
        userIdentifier: String,
        context: Context,
        rewardListener: TRRewardCallback? = null,
        errorListener: TRErrorCallback,
        sdkReadyListener: TRSdkReadyCallback,
        quickQuestionListener: TRQQDataCallback? = null,
        initOptions: TapInitOptions? = null
    ) {
        this.initializationResponseReceived = false
        this.initOptions = initOptions
        this.userIdentifier = userIdentifier
        this.userErrorListener = errorListener
        this.userSdkReadyListener = sdkReadyListener
        when (sdkStateHolder.state) {
            SdkState.initialized -> {
                LogUtils.internal(LOG_TAG, "Already initialized.")
                orchestrator?.resetRewardCallback(rewardListener)
                orchestrator?.notifyRewardCallbackAvailability(rewardListener)

                orchestrator?.resetQQDataCallback(quickQuestionListener)
                orchestrator?.notifyQQDataCallbackAvailability(quickQuestionListener)
                internalSdkReadyListener.onTapResearchSdkReady()
                TimerUtil.stopTimer(INITIALIZATION_TIMER)
            }

            SdkState.initializing -> {
                LogUtils.internal(LOG_TAG, "Initialization already in progress..")
            }

            else -> { // state 'not_started_yet' or 'failed' from previous attempt
                LogUtils.internal(LOG_TAG, "Initializing TapResearch SDK..")
                startInitializationTimer()
                ScreenONReceiver.register(context)
                try {
                    (context.applicationContext as Application).registerActivityLifecycleCallbacks(
                        lifecycleObserver,
                    )
                    (context.applicationContext as Application).registerComponentCallbacks(
                        lifecycleObserver,
                    )
                } catch (e: Exception) {
                    LogUtils.internal(LOG_TAG, "Could not register ActivityLifecycleCallbacks: $e")
                }
                try {
                    ProcessLifecycleOwner.get().lifecycle.addObserver(lifecycleObserver)
                } catch (e: Exception) {
                    LogUtils.internal(LOG_TAG, "Could not addObserver to ProcessLifecycleOwner: $e")
                }
            }
        }

        configure(
            apiToken = apiToken,
            userIdentifier = userIdentifier,
            applicationContext = context.applicationContext,
            rewardListener = rewardListener,
            errorListener = internalErrorListener,
            sdkReadyListener = internalSdkReadyListener,
            quickQuestionListener = quickQuestionListener,
        )
    }

    /**
     * Sets the unique user identifier in TapResearch.  Can cause more placements to become
     * available.
     *
     * @param userIdentifier The unique user identifier.
     */
    fun setUserIdentifier(userIdentifier: String) {
        this.userIdentifier = userIdentifier
        ParameterStorage.saveUserIdentifier(userIdentifier)
        executeOrcaReadyTask(
            jobName = "setUserIdentifier",
            orcaReadyTask = {
                orchestrator?.evaluateJavascript("updateCurrentUser('$userIdentifier')")
            }
        )
    }

    private fun canShowContentForPlacementPrivate(tag: String, errorCallback: TRErrorCallback, caller: String?): Boolean {
        // sdk could be in the middle of initializing or re-initializing from a dead orca connection so don't bow out just yet!
        if (sdkStateHolder.state == SdkState.not_started_yet || sdkStateHolder.state == SdkState.initialization_failed) {
            errorCallback.onTapResearchDidError(
                TRError(
                    code = TapErrorCodes.NOT_INITIALIZED.code,
                    description = TapErrorCodes.NOT_INITIALIZED.errorMessage() + " -> [${sdkStateHolder.state}]",
                ),
            )
            ShowContentLogger.log("canShowContentForPlacement","sdk not ready. caller: ${caller?:"direct"}. sdk-state: ${sdkStateHolder.state}. placement: $tag.")
            return false
        }

        application?.let {
            if (ConnectionUtils.isOnline(it).not()) {
                errorCallback.onTapResearchDidError(
                    TRError(
                        code = TapErrorCodes.NETWORK_ERROR.code,
                        description = TapErrorCodes.NETWORK_ERROR.errorMessage(),
                    ),
                )
                LogUtils.e(LOG_TAG, "canShowContentForPlacement: No internet connection")
                return false
            }
        }

        val placement = PlacementStorage.getPlacementByTag(tag)
        if (placement.error != null) {
            errorCallback.onTapResearchDidError(placement.error)
            ShowContentLogger.log("canShowContentForPlacement","placement error: ${placement.error}. caller: ${caller?:"direct"}. placement: $tag.")
            return false
        }
        return true
    }

    /**
     * Immediately indicates whether survey wall content can be shown for the given placement.
     *
     * @param tag The placement tag string.  E.g. "home-screen" or "earn-center"
     * @param errorCallback Listens for any errors that might occur during this call.
     * @return True if content for the given placement can be shown.  False otherwise.
     */
    fun canShowContentForPlacement(tag: String, errorCallback: TRErrorCallback): Boolean {
        return canShowContentForPlacementPrivate(tag, errorCallback, null)
    }

    internal fun runManualInitialization() {
        if (initOptions?.userAttributes != null) {
            val userAttributesJson = (initOptions?.userAttributes as Map<*, *>?)?.let { JSONObject(it) }
            val escapedUserAttributesJson = userAttributesJson.toString().replace("'", "\\'")
            LogUtils.internal(LOG_TAG, "Sending user attributes: $userAttributesJson")

            orchestrator?.evaluateJavascript("sendUserAttributes('$escapedUserAttributesJson', ${initOptions?.clearPreviousAttributes ?: false})")
        }
    }

    /**
     * Sends user attributes to TapResearch.
     *
     * @param userAttributes The user attributes map that contains your targeting parameters.
     * @param clearPreviousAttributes If true, will clear the previously sent user attributes.
     * @param errorCallback Listens for any errors that might occur during this call.
     *
     */
    fun sendUserAttributes(
        userAttributes: HashMap<String, Any>,
        clearPreviousAttributes: Boolean? = false,
        errorCallback: TRErrorCallback,
    ) {
        val userAttributesJson = (userAttributes as Map<*, *>?)?.let { JSONObject(it) }

        if (userIdentifier.isEmpty()) {
            LogUtils.e(LOG_TAG, TapErrorCodes.MISSING_USER_IDENTIFIER.errorMessage())
            errorCallback.onTapResearchDidError(
                TRError(
                    code = TapErrorCodes.MISSING_USER_IDENTIFIER.code,
                    description = TapErrorCodes.MISSING_USER_IDENTIFIER.errorMessage(),
                ),
            )
            return
        } else {
            // Escape the JSON string by replacing apostrophes with the escaped version
            val escapedUserAttributesJson = userAttributesJson.toString().replace("'", "\\'")
            LogUtils.internal(LOG_TAG, "Sending user attributes: $escapedUserAttributesJson")

            executeOrcaReadyTask(
                jobName = "sendUserAttributes",
                orcaReadyTask = {
                    if (orcaCanRespond) {
                        orchestrator?.apply {
                            sendUserAttributesErrorCallback = errorCallback
                            evaluateJavascript("sendUserAttributes('$escapedUserAttributesJson', $clearPreviousAttributes)")
                        }
                    } else {
                        CoroutineScope(Dispatchers.Main).launch {
                            try {
                                errorCallback.onTapResearchDidError(
                                    TRError(
                                        TapErrorCodes.NOT_INITIALIZED.code,
                                        TapErrorCodes.NOT_INITIALIZED.description,
                                    ),
                                )
                            } catch (throwable: Throwable) {
                                RemoteEventLogger.postEvent(
                                    REMOTE_LOG_LEVEL,
                                    "TapResearch.sendUserAttributes",
                                    "exception occurred. internal state: ${sdkStateHolder.state}",
                                    throwable
                                )
                            }
                        }
                    }
                }
            )
        }
    }

    /**
     * Show TapResearch content activity for a given placement tag.
     *
     * @param tag The placement tag string.  E.g. "home-screen" or "earn-center"
     * @param contentListener Listens for content events such as content shown and dismissed.
     * @param customParameters Optional custom targeting parameters.
     * @param errorListener Listens for any errors that might occur during this call.
     */
    fun showContentForPlacement(
        tag: String,
        contentListener: TRContentCallback? = null,
        customParameters: HashMap<String, Any>? = null,
        errorListener: TRErrorCallback,
    ) {
        showContentForPlacement(tag, contentListener, customParameters, errorListener, null)
    }

    /**
     * Show TapResearch content activity for a given placement tag.
     *
     * @param tag The placement tag string.  E.g. "home-screen" or "earn-center"
     * @param contentCallback Listens for content shown and dismissed events.
     * @param customParameters Optional custom targeting parameters.
     * @param errorCallback Listens for any errors that might occur during this call.
     * @param optionalParentActivity Specify optional parent activity.  Not recommended unless
     * you have a unique situation where you want to show a survey from the device's lock screen.
     */
    fun showContentForPlacement(
        tag: String,
        contentCallback: TRContentCallback? = null,
        customParameters: HashMap<String, Any>? = null,
        errorCallback: TRErrorCallback,
        optionalParentActivity: Activity? = null,
    ) {
        // No async showContent if canShowContent is false
        if (!canShowContentForPlacementPrivate(tag, errorCallback, "showContentForPlacement")) return

        CoroutineScope(Dispatchers.IO).launch {
            // could be initializing or re-initializing from a dead orca connection
            if (sdkStateHolder.state == SdkState.initializing) {
                waitWhileInitializing()
                if (sdkStateHolder.state != SdkState.initialized) {
                    errorCallback.onTapResearchDidError(
                        TRError(
                            code = TapErrorCodes.NOT_INITIALIZED.code,
                            description = TapErrorCodes.NOT_INITIALIZED.errorMessage() + " -> [${sdkStateHolder.state}]",
                        ),
                    )
                    // Should be fairly rare as it shouldn't take the SDK 20 seconds to initialize
                    ShowContentLogger.log("showContentForPlacement","sdk took too long to initialize. sdk-state: ${sdkStateHolder.state}. placement: $tag.")
                    return@launch
                }
            }
            (Dispatchers.Main) {
                if (canShowContentForPlacementPrivate(tag, errorCallback, "showContentForPlacement")) {
                    orchestrator?.let { orchestrator ->
                        orchestrator.contentCallback = contentCallback
                        orchestrator.showContentForPlacementErrorCallback = errorCallback
                        activityWeakReference = if (optionalParentActivity != null) WeakReference(optionalParentActivity) else null
                        TapResearch.apply {
                            currentPlacement = PlacementStorage.getPlacementByTag(tag)
                            val customParametersJson = convertToAny(customParameters)
                            orchestrator.evaluateJavascript("showContentForPlacement('$tag', '$customParametersJson')")
                        }
                    }
                }
            }
        }
    }

    @Deprecated(
        message = "No longer need application parameter",
        replaceWith = ReplaceWith("showContentForPlacement(tag, contentCallback, customParameters, errorCallback)"),
        level = DeprecationLevel.WARNING,
    )
    fun showContentForPlacement(
        tag: String,
        application: Application,
        contentListener: TRContentCallback? = null,
        customParameters: HashMap<String, Any>? = null,
        errorListener: TRErrorCallback,
    ) {
        showContentForPlacement(tag, contentListener, customParameters, errorListener)
    }

    private fun presentationCallback(configuration: String) {
        portraitConfiguration = null
        landscapeConfiguration = null

        configurationString = configuration

        // Deserialize JSON into TRConfiguration class
        currentOfferConfiguration = json.decodeFromString<TRConfiguration>(configurationString)

        // Only parse as event configuration if the first JSON parse did not return TRConfiguration (checked via the .webview property existing)
        val currentOfferEventConfiguration = if (currentOfferConfiguration?.webview !== null) {
            null
        } else {
            json.decodeFromString<OfferEventConfiguration>(
                configurationString,
            )
        }

        if (currentOfferEventConfiguration != null) {
            // Set portrait and landscape configurations
            if (currentOfferEventConfiguration.portrait != null) {
                portraitConfiguration = currentOfferEventConfiguration.portrait
            }
            if (currentOfferEventConfiguration.landscape != null) {
                landscapeConfiguration = currentOfferEventConfiguration.landscape
            }
        }
        presentContentWithConfiguration()
    }

    private fun configure(apiToken: String, userIdentifier: String, applicationContext: Context, rewardListener: TRRewardCallback? = null, errorListener: TRErrorCallback, sdkReadyListener: TRSdkReadyCallback, quickQuestionListener: TRQQDataCallback? = null) {
        ParameterStorage.saveApiToken(apiToken)
        ParameterStorage.saveUserIdentifier(userIdentifier)

        if (orchestrator == null) {
            orchestrator = TROrchestrator(
                apiToken = apiToken,
                userIdentifier = userIdentifier,
                applicationContext = applicationContext,
                presentationCallback = { configuration -> presentationCallback(configuration) },
                rewardCallback = rewardListener,
                sdkReadyCallback = sdkReadyListener,
                tapDataCallback = quickQuestionListener,
                isManualInit = initOptions?.userAttributes != null,
                errorCallback = errorListener,
                sdkStateHolder = sdkStateHolder,
            )
        } else {
            orchestrator!!.resetCallbacks(
                presentationCallback = { configuration -> presentationCallback(configuration) },
                rewardCallback = rewardListener,
                sdkReadyCallback = sdkReadyListener,
                tapDataCallback = quickQuestionListener,
                errorCallback = errorListener,
            )
        }
    }

    internal fun setOrchestratorWebViewCallback(webViewCallback: ((webViewState: TRWebViewState) -> Unit)?) {
        orchestrator?.webViewCallback = webViewCallback
    }

    internal fun childWebViewMessage(): Message? {
        return orchestrator?.webViewMessage
    }

    private fun presentContentWithConfiguration() {
        // Run code on the UI thread
        CoroutineScope(Dispatchers.Main).launch {
            if (currentOfferConfiguration?.webview !== null || currentOfferConfiguration?.navigationBar?.show == 0) {
                startTRWebViewActivity()
            } else {
                startDialogActivity()
            }
        }
    }

    private fun startTRWebViewActivity() {
        wallWebViewIntent = Intent(application, TRWebViewActivity::class.java)
        wallWebViewIntent?.putExtra("configuration", configurationString)
        wallWebViewIntent?.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
        if (!(BuildConfig.DEBUG && doTestSkipWall)) {
            application?.startActivity(wallWebViewIntent)
        }
    }

    private fun activityIsValid(): Boolean {
        activityWeakReference?.apply {
            this.get()?.let { activity ->
                if (!activity.isDestroyed) {
                    return true
                }
            }
        }
        return false
    }

    private fun startDialogActivity() {
        if (activityIsValid()) {
            activityWeakReference?.get()?.let {
                trDialogNonActivity = TRDialogNonActivity { trDialogNonActivity = null }.apply {
                    startFauxActivity(it, portraitConfiguration, landscapeConfiguration)
                }
            }
        } else {
            interstitialIntent = Intent(application, TRDialogActivity::class.java)
            portraitConfiguration?.let { conf ->
                interstitialIntent?.putExtra(
                    "portraitConfiguration",
                    json.encodeToString(TRConfiguration.serializer(), conf),
                )
            }
            landscapeConfiguration?.let { conf ->
                interstitialIntent?.putExtra(
                    "landscapeConfiguration",
                    json.encodeToString(TRConfiguration.serializer(), conf),
                )
            }
            interstitialIntent?.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            if (!(BuildConfig.DEBUG && doTestSkipWall)) {
                application?.startActivity(interstitialIntent)
            }
        }
    }

    /**
     * Optional!  Only invoke when you are sure the user wants to quit the entire app
     * and remove all internal caching.  Not detrimental if not used.
     * Do not call from Activity.onDestroy().  After calling, the next initialize call
     * will be slow.  Does not improve memory consumption.
     */
    fun quitAndCleanup() {
        orchestrator?.dialogWebView?.destroy()
        orchestrator?.surveyWebView?.destroy()
        orchestrator?.orcaWebView?.destroy()
        orchestrator = null
        sdkStateHolder.state = SdkState.not_started_yet
        initOptions = null
        currentPlacement = null
        currentOfferConfiguration = null
        configurationString = ""
        portraitConfiguration = null
        landscapeConfiguration = null
        orcaCanRespond = false
        userIdentifier = ""
        activityWeakReference = null
        application?.let {
            try {
                (it.applicationContext as Application).unregisterActivityLifecycleCallbacks(
                    lifecycleObserver,
                )
                (it.applicationContext as Application).unregisterComponentCallbacks(
                    lifecycleObserver,
                )
            } catch (e: Exception) {
                LogUtils.internal(LOG_TAG, "Could not unregister lifecycleObserver: $e")
            }
        }
        try {
            ProcessLifecycleOwner.get().lifecycle.removeObserver(lifecycleObserver)
        } catch (e: Exception) {
            LogUtils.internal(LOG_TAG, "Could not remove ProcessLifecycleOwner: $e")
        }
        application = null
        appInForeground = false
        currentPlacement = null
        webviewTypeOpen = WebViewTypeOpen.NONE
    }

    // Comment out of the official public SDK - only for diagnostic and hardening purposes!
    /*
    fun kill(inSeconds: Int? = 5, doSetWebViewToNull: Boolean = false, doCrashWebView: Boolean = false) {
        if (BuildConfig.DEBUG) {
            CoroutineScope(Dispatchers.IO).launch {
                LogUtils.internal(LOG_TAG, "waiting to kill orca in $inSeconds...")
                delay(1000L*(inSeconds?:5))
                (Dispatchers.Main) {
                    if (doSetWebViewToNull) {
                        LogUtils.internal(LOG_TAG, "doSetWebViewToNull...")
                        // artificially set orca webview to null
                        orchestrator?.orcaWebView?.also {
                            orchestrator?.orcaWebView = null
                        }
                    } else if (doCrashWebView) {
                        LogUtils.internal(LOG_TAG, "doCrashWebView...")
                        // artificially trigger orca webview.onRenderProcessGone
                        orchestrator?.orcaWebView?.loadUrl("chrome://crash")
                    } else {
                        LogUtils.internal(LOG_TAG, "Orca.die(1)...")
                        // artificially trigger orca death via js
                        orchestrator?.evaluateJavascript("Orca.die(1)")
                    }
                }
            }
        }
    }
    */

    internal fun sdkState(): SdkState {
        return sdkStateHolder.state
    }

    internal fun onOrcaCanRespond() {
        if (!orcaCanRespond) {
            orchestrator?.storeTag()
        }
        orcaCanRespond = true
    }

    /**
     * Immediately returns true if TapResearch SDK was successfully initialized and is ready to be used.
     * False, otherwise.
     */
    fun isReady(): Boolean {
        return sdkStateHolder.state == SdkState.initialized
    }

    /**
     * If return true, indicates TapResearch SDK is currently in the process
     * of initializing.
     * If false, indicates SDK initialization process has either not started
     * yet OR the process has failed due to bad network or some other issue.
     */
    internal fun isInitializing(): Boolean {
        return sdkStateHolder.state == SdkState.initializing
    }

    /**
     * Set surveys refreshed listener.
     *
     * @param surveysRefreshedListener Listens for when native preview wall placements are
     * refreshed.  The refreshed placement tag can then be queried by calling getSurveysForPlacement.
     * Passing null will un-listen and could be called in onPause() lifecycle events, if desired.
     * @see getSurveysForPlacement
     *
     */
    fun setSurveysRefreshedListener(surveysRefreshedListener: TRSurveysRefreshedListener?) {
        orchestrator?.trjsInterface?.surveysRefreshedListener = surveysRefreshedListener
    }

    /**
     * Immediately indicates if native preview wall surveys are available for a given placement.
     *
     * @param placementTag The placement tag string.  E.g. "home-screen" or "earn-center"
     * @param errorListener Listens for any errors that might occur during this call.
     * @return true if native surveys are available for given placement.
     *
     * @see getSurveysForPlacement
     */
    fun hasSurveysForPlacement(placementTag: String,
                               errorListener: TRErrorCallback): Boolean {
        return !getSurveysForPlacement(placementTag, errorListener).isNullOrEmpty()
    }

    /**
     * Immediately returns a list of cached native preview wall surveys for the given placement.
     *
     * @param placementTag - The placement tag string.  E.g. "home-screen" or "earn-center"
     * @param errorListener - Listens for any errors that might occur during this call.
     * @return an immediate list of cached native preview wall surveys.  null, if none are available.
     *
     * @see showSurveyForPlacement
     */
    fun getSurveysForPlacement(placementTag: String,
                               errorListener: TRErrorCallback): List<TRSurvey>? {
        if (!canShowContentForPlacementPrivate(placementTag, errorListener, "getSurveysForPlacement")) {
            return null
        }
        orchestrator?.evaluateJavascript("onGetSurveysImpression('$placementTag')")
        return PlacementStorage.getPlacementByTagOrNull(placementTag)?.offer?.surveys?.toList()
    }

    /**
     * Show a native preview wall survey for the given survey id AND placement.  Survey id parameter
     * was obtained via getSurveysForPlacement.
     *
     * @param placementTag The placement tag string.  E.g. "home-screen" or "earn-center"
     * @param surveyId Survey id of the native preview wall survey to be shown.
     * @param customParameters Optional custom targeting parameters.
     * @param contentListener Optional; Listens for content shown and dismiss events.
     * @param errorListener Listens for any errors that might occur during this call.
     *
     * @see getSurveysForPlacement
     */
    fun showSurveyForPlacement(
                                placementTag: String,
                                surveyId: String,
                                customParameters: HashMap<String, Any>? = null,
                                contentListener: TRContentCallback? = null,
                                errorListener: TRErrorCallback,
                               ) {

        CoroutineScope(Dispatchers.IO).launch {
            // could be initializing or re-initializing from a dead orca connection
            if (sdkStateHolder.state == SdkState.initializing) {
                waitWhileInitializing()
            }
            (Dispatchers.Main) {
                if (canShowContentForPlacementPrivate(placementTag, errorListener, "showSurveyForPlacement")) {
                    orchestrator?.let { orchestrator ->
                        orchestrator.contentCallback = contentListener
                        orchestrator.showContentForPlacementErrorCallback = errorListener
                        TapResearch.apply {
                            currentPlacement = PlacementStorage.getPlacementByTag(placementTag)
                            val customParametersJson = convertToAny(customParameters)
                            orchestrator.evaluateJavascript("showSurveyForPlacement('$placementTag', '$surveyId', '$customParametersJson')")
                        }
                    }
                }
            }
        }
    }

    private fun convertToAny(customParameters: HashMap<String, Any>? = null): Any {
        return customParameters?.let { cp ->
            (cp as Map<*, *>?)?.let { JSONObject(it) }
        } ?: ""
    }
}
