package com.tapresearch.tapsdk.utils

import com.tapresearch.tapsdk.TapResearch.application
import com.tapresearch.tapsdk.TapResearch.orcaCanRespond
import com.tapresearch.tapsdk.TapResearch.sdkStateHolder
import com.tapresearch.tapsdk.state.SdkState
import com.tapresearch.tapsdk.utils.TapConstants.LOG_TAG
import com.tapresearch.tapsdk.utils.TapConstants.REMOTE_LOG_LEVEL
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.serialization.InternalSerializationApi

@InternalSerializationApi
internal object TRHelper {

    private val jobMap = HashMap<String,Job>()

    internal fun executeOrcaReadyTask(
        jobName: String,
        orcaReadyTask: (() -> Unit),
    ) {
        //cancel any previous jobs that may be waiting for orca
        synchronized(this@TRHelper) {
            var job = jobMap[jobName]
            job?.apply {
                if (isActive) {
                    cancel()
                    LogUtils.internal(LOG_TAG, "job: [$job] -> $jobName was cancelled; replaced by new job. orcaCanRespond: $orcaCanRespond")
                }
                jobMap.remove(jobName)
            }
            job = CoroutineScope(Dispatchers.IO).launch {
                try {
                    waitUntilOrcaCanRespond()
                    LogUtils.internal(LOG_TAG, "job: [$job] -> $jobName waited for orca. orcaCanRespond: $orcaCanRespond")
                    if (orcaCanRespond) {
                        orcaReadyTask()
                    }
                    LogUtils.internal(LOG_TAG,"job: [$job] -> $jobName completed. orcaCanRespond: $orcaCanRespond")
                } catch (throwable: Throwable) {
                    RemoteEventLogger.postEvent(REMOTE_LOG_LEVEL, "TapResearch.executeOrcaReadyTask", "job: [$jobName] exception. sdkState: [${sdkStateHolder.state}]", throwable)
                }
                synchronized(this@TRHelper) {
                    jobMap.remove(jobName)
                    LogUtils.internal(LOG_TAG, "job:[$job] -> $jobName removed from cache. orcaCanRespond: $orcaCanRespond")
                }
            }
            jobMap[jobName] = job
            LogUtils.internal(LOG_TAG, "running job:[$job] -> $jobName orcaCanRespond: $orcaCanRespond")
        }
    }

    private suspend fun waitUntilOrcaCanRespond() {
        var isOnline = true
        application?.let {
            isOnline = ConnectionUtils.isOnline(it)
        }
        var count = 20000 // max 20 seconds; more than enough for slow connections
        while (count > 0 && !orcaCanRespond && isOnline) {
            delay(100)
            count -= 100
        }
        LogUtils.internal(
            LOG_TAG,
            "orcaCanRespond: $orcaCanRespond. $count millis was remaining to wait.",
        )
    }

    internal suspend fun waitWhileInitializing() {
        var count = 20000 // max 20 seconds; more than enough for slow connections
        while (count > 0 && sdkStateHolder.state == SdkState.initializing) {
            delay(100)
            count -= 100
        }
        LogUtils.internal(LOG_TAG, "initialized: $count millis was remaining to wait.")
    }
}
