package com.tapresearch.tapsdk.utils

import com.tapresearch.tapresearchkotlinsdk.BuildConfig
import com.tapresearch.tapsdk.TapResearch
import com.tapresearch.tapsdk.storage.ParameterStorage
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.serialization.InternalSerializationApi
import org.json.JSONObject
import java.io.PrintWriter
import java.io.StringWriter

@InternalSerializationApi
internal object RemoteEventLogger {

    private const val TAG = "TRLog_REL"
    private const val PROD_EVENT_END_POINT = "https://api.tapresearch.com/supply_api/v4/events"
    private const val DEBUG_EVENT_END_POINT = "https://hoth.tapresearch.com/supply_api/v4/events"

    /**
     *     logLevel: 4,
     *     platformType: 'sdk',
     *     tag: <some event id (e.g. 'ClassName#methodName')>,
     *     message: <event description (e.g. Orca failed to initialize|respond)>
     */
    fun postEvent(
        logLevel: Int,
        tag: String,
        message: String,
        throwable: Throwable? = null,
        logCategory: String? = null,
        logCategoryDetail: String? = null,
    ) {
        CoroutineScope(Dispatchers.IO).launch {
            internalPostEvent(logLevel, tag, message, throwable, logCategory, logCategoryDetail)
        }
    }

    private fun internalPostEvent(
        logLevel: Int,
        tag: String,
        message: String,
        throwable: Throwable? = null,
        logCategory: String? = null,
        logCategoryDetail: String? = null,
    ) {
        val endPoint = if (BuildConfig.DEBUG) DEBUG_EVENT_END_POINT else PROD_EVENT_END_POINT
        var finalMessage = "$message "
        throwable?.let {
            val stringWriter = StringWriter()
            it.printStackTrace(PrintWriter(stringWriter))
            val exceptionAsString = stringWriter.toString()
            finalMessage += exceptionAsString
        }
        try {
            val httpHelper = HttpHelper(endPoint, ParameterStorage.refreshHeaders())
            val eventPayload = JSONObject()
            eventPayload.put("log_level", logLevel)
            eventPayload.put("platform_type", "sdk")
            eventPayload.put("tag", tag)
            eventPayload.put("active_placement_tag", TapResearch.currentPlacement?.tag?:"none")
            eventPayload.put("app_in_foreground", TapResearch.appInForeground)
            eventPayload.put("webview_type_open", TapResearch.webviewTypeOpen.parameterValue)
            eventPayload.put("message", finalMessage)
            logCategory?.apply { eventPayload.put("log_category", this) }
            logCategoryDetail?.apply { eventPayload.put("log_category_detail", this) }
            val response = httpHelper.post(eventPayload.toString())
            LogUtils.internal(
                TAG,
                "posted remote event to $endPoint response: $response eventPayload: $eventPayload"
            )
        } catch (t: Throwable) {
            LogUtils.internal(TAG, "internalPostEvent failed: $t")
        }
    }
}
