package com.tapresearch.tapsdk.models

import androidx.annotation.Keep
import org.json.JSONObject
import java.io.Serializable

@Keep
internal class PlacementCustomParameters : Serializable {

    private val parameterList: MutableList<PlacementParameter> = ArrayList()

    @Keep
    class PlacementCustomParametersException(message: String) : Exception(message)

    data class PlacementParameter(
        val key: String,
        val value: String,
    ) {
        class Builder {
            private var key: String? = null
            private var value: String? = null

            fun key(key: String): Builder {
                this.key = key
                return this
            }

            fun value(value: String): Builder {
                this.value = value
                return this
            }

            @Throws(PlacementCustomParametersException::class)
            fun build(): PlacementParameter {
                validatePlacementPassValues()
                return PlacementParameter(key!!, value!!)
            }

            @Throws(PlacementCustomParametersException::class)
            private fun validatePlacementPassValues() {
                when {
                    key == null -> {
                        throw PlacementCustomParametersException("The parameter key can't be null")
                    }

                    key!!.isEmpty() -> {
                        throw PlacementCustomParametersException("The parameter key size can't be zero")
                    }

                    value == null -> {
                        throw PlacementCustomParametersException("The parameter value can't be null")
                    }

                    value!!.length > MAX_VALUE_LENGTH -> {
                        throw PlacementCustomParametersException(
                            String.format(
                                "The parameter value " +
                                    "length should be less than %d characters",
                                MAX_VALUE_LENGTH,
                            ),
                        )
                    }
                }
            }
        }
    }

    @Throws(PlacementCustomParametersException::class)
    fun addParameter(parameter: PlacementParameter) {
        if (parameterList.size < MAX_PASS_VALUES) {
            parameterList.add(parameter)
        } else {
            throw PlacementCustomParametersException(
                String.format(
                    "Maximum number of " +
                        "parameters exceeded (%d)",
                    MAX_PASS_VALUES,
                ),
            )
        }
    }

    @Keep
    private fun toMap(): HashMap<String, String> {
        val map = HashMap<String, String>()
        for (parameter in parameterList) {
            map[parameter.key] = parameter.value
        }
        return map
    }

    @Keep
    fun toJson(): JSONObject {
        return JSONObject(this.toMap() as Map<String, String>)
    }

    @Throws(PlacementCustomParametersException::class)
    @Keep
    fun fromMap(map: Map<String, String>): PlacementCustomParameters {
        if (map.size > MAX_PASS_VALUES) {
            throw PlacementCustomParametersException(
                String.format(
                    "The maximum number of " +
                        "parameters is %d",
                    MAX_PASS_VALUES,
                ),
            )
        } else {
            val customParameters = PlacementCustomParameters()
            for ((key, value) in map) {
                val placementParameter = PlacementParameter.Builder()
                    .key(key)
                    .value(value).build()
                customParameters.addParameter(placementParameter)
            }
            return customParameters
        }
    }

    companion object {
        internal const val MAX_PASS_VALUES = 5
        internal const val MAX_VALUE_LENGTH = 256
    }
}
